<?php declare(strict_types = 1);

namespace DynamicModule\Model\Services;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Event\InstagramMediaEvent;
use Core\Model\Helpers\Strings;
use Core\Model\Lang\Langs;
use DynamicModule\FrontModule\Model\Repository\Groups;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Field;
use DynamicModule\Model\Entities\Group;
use DynamicModule\Model\Entities\GroupMember;
use DynamicModule\Model\Entities\Member;
use DynamicModule\Model\Repository\Members;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Utils\DateTime;
use Nette\Utils\Json;
use Tracy\Debugger;

class MultihubInstagramService
{
	protected EventDispatcher        $eventDispatcher;
	protected Members                $members;
	protected EntityManagerDecorator $em;
	protected Storage                $cacheStorage;
	protected Langs                  $langsService;

	public function __construct(
		EventDispatcher        $eventDispatcher,
		Members                $members,
		EntityManagerDecorator $em,
		Storage                $cacheStorage,
		Langs                  $langsService
	)
	{
		$this->eventDispatcher = $eventDispatcher;
		$this->members         = $members;
		$this->em              = $em;
		$this->cacheStorage    = $cacheStorage;
		$this->langsService    = $langsService;
	}

	public function downloadLatestMedia(): void
	{

		foreach (DynamicModuleConfig::load('multihubInstagramMap') ?? [] as $config) {
			$current   = [];
			$apiMedia  = [];
			$forUpdate = [];

			foreach ($this->members->getQueryBuilderByGroup($config['groupId'])
				         ->getQuery()->getResult() as $row) {
				/** @var GroupMember $row */
				$v                                     = $row->member->getFieldsValues();
				$current[$v[$config['idMemberField']]] = $v;
			}

			$curl = curl_init();

			curl_setopt_array($curl, [
				CURLOPT_URL            => 'https://social.multihub.cz/api/v1/instagram/' . $config['api']['profileId'] . '/latest-media',
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_FOLLOWLOCATION => true,
				CURLOPT_CUSTOMREQUEST  => 'GET',
				CURLOPT_HTTPHEADER     => [
					'X-Account: ' . $config['api']['accountId'],
					'X-Token: ' . $config['api']['token'],
				],
			]);

			$response = curl_exec($curl);
			curl_close($curl);

			$groupId = $config['groupId'];

			try {
				foreach (Json::decode((string) $response, Json::FORCE_ARRAY)['data'] as $row) {
					if (isset($current[$row['mediaId']])) {
						$forUpdate[$row['mediaId']] = $row;
						continue;
					}

					$apiMedia[$row['mediaId']] = $row;
				}
			} catch (\Exception $e) {
				Debugger::log($e, 'multihub_instagram');
				throw $e;
			}

			$cache       = new Cache($this->cacheStorage, Groups::CACHE_NAMESPACE);
			$memberCache = new Cache($this->cacheStorage, Members::CACHE_NAMESPACE);

			foreach ($forUpdate as $mediaId => $row) {
				/** @var GroupMember|null $groupMember */
				$groupMember = $this->members->getQueryBuilderByGroup($groupId)
					->innerJoin('gm.member', 'm')
					->innerJoin('m.fields', 'f')
					->andWhere('f.key = :fKey AND f.value = :fValue')
					->setParameters([
						'fKey'   => $config['idMemberField'],
						'fValue' => $mediaId,
					])
					->getQuery()
					->getOneOrNullResult();

				if (!$groupMember) {
					continue;
				}

				$title  = self::parseValue($config['titleField'], $row);
				$title  = Strings::truncate($title, 255, '');
				$member = $groupMember->member;

				$member->title = $title;
				$this->em->persist($member);

				$fields = $member->getFieldsByKey();

				foreach ($config['fields'] ?? [] as $kField => $vField) {
					if (isset($fields[$kField])) {
						$fields[$kField]->value = self::parseValue($vField, $row);
						$this->em->persist($fields[$kField]);
						continue;
					}

					$field = new Field($kField, self::parseValue($vField, $row));

					$this->em->persist($field);
					$member->fields->add($field);
				}

				$this->em->flush();

				foreach ($this->langsService->getLangs(false) as $lang) {
					$cache->remove($config['module'] . '_' . md5((string) $groupId) . '_' . $lang->getTag());

					$memberCache->remove('dynamicModuleMember/' . $lang->getTag() . '/' . $member->getId());
				}
				$cache->clean([Cache::TAGS => [Groups::CACHE_NAMESPACE]]);

				$cache = new Cache($this->cacheStorage, Members::CACHE_NAMESPACE);
				$cache->clean([Cache::TAGS => Members::CACHE_NAMESPACE]);

				$event = new InstagramMediaEvent((string) $config['api']['profileId'], $row['mediaId'], $member, $row);
				$this->eventDispatcher->dispatch($event, 'dynamicModule.instagramMedia::updated');
			}

			foreach (array_reverse($apiMedia) as $row) {
				$title = self::parseValue($config['titleField'], $row);

				$title = Strings::truncate($title, 255, '');

				$member = new Member((string) $title, $config['template']);

				/** @var Group $group */
				$group = $this->em->getRepository(Group::class)->find($groupId);
				$member->addGroup($group);

				foreach ($config['fields'] ?? [] as $kField => $vField) {
					$field = new Field($kField, self::parseValue($vField, $row));
					$this->em->persist($field);

					$member->fields->add($field);
				}

				$this->em->persist($member);
				$this->em->flush();

				foreach ($this->langsService->getLangs(false) as $lang) {
					$cache->remove($config['module'] . '_' . md5((string) $groupId) . '_' . $lang->getTag());

					$memberCache->remove('dynamicModuleMember/' . $lang->getTag() . '/' . $member->getId());
				}
				$cache->clean([Cache::TAGS => [Groups::CACHE_NAMESPACE]]);

				$cache = new Cache($this->cacheStorage, Members::CACHE_NAMESPACE);
				$cache->clean([Cache::TAGS => Members::CACHE_NAMESPACE]);

				$event = new InstagramMediaEvent((string) $config['api']['profileId'], $row['mediaId'], $member, $row);
				$this->eventDispatcher->dispatch($event, 'dynamicModule.instagramMedia::created');
			}
		}
	}

	public static function parseValue(string $key, array $data): ?string
	{
		$result = null;
		if (Strings::contains($key, ':')) {
			$tmp = explode(':', $key, 2);

			if ($tmp[1] === 'firstLine') {
				$result = explode("\n", (string) $data[$tmp[0]])[0] ?? null;
			} else if ($tmp[1] === 'skipFirstLine') {
				$result = explode("\n", (string) $data[$tmp[0]], 2)[1] ?? null;
			}

			$result = trim((string) $result);
		} else {
			$result = $data[$key] ?? null;
		}

		if ($key === 'timestamp') {
			$datetime = DateTime::createFromFormat("Y-m-d\TH:i:sO", $result);
			$result   = $datetime->format('Y-m-d H:i:s');
		} else {
			$result = trim((string) $result);

			if (Strings::startsWith($key, 'caption')) {
				$result = html_entity_decode($result, ENT_QUOTES | ENT_HTML5, 'UTF-8');
				$result = mb_convert_encoding($result, 'UTF-8', 'UTF-8');
				$result = preg_replace('/[^\x{0000}-\x{FFFF}]+/u', '', $result);

				$result = nl2br($result);
			}
		}

		return (string) $result;
	}
}
