<?php declare(strict_types = 1);

namespace DynamicModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Lang\Langs;
use DynamicModule\FrontModule\Model\Repository\Groups;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Field;
use DynamicModule\Model\Entities\Group;
use DynamicModule\Model\Entities\Member;
use DynamicModule\Model\Repository\Members;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class GoogleSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected Storage                $cacheStorage;
	protected Langs                  $langsService;

	public function __construct(
		EntityManagerDecorator $em,
		Storage                $cacheStorage,
		Langs                  $langsService
	)
	{
		$this->em           = $em;
		$this->cacheStorage = $cacheStorage;
		$this->langsService = $langsService;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'google.googleReview::created' => 'googleReviewCreated',
		];
	}

	/**
	 * @param object|\Google\Model\Events\GoogleReviewEvent $event
	 *
	 * @return void
	 */
	public function googleReviewCreated(object $event): void
	{
		if (!is_array(DynamicModuleConfig::load('googleReviewCreateMap'))) {
			return;
		}

		foreach (DynamicModuleConfig::load('googleReviewCreateMap') as $v) {
			$groupId = $v['groupId'];

			$title = $event->googleReview->{$v['titleField']};

			$member = new Member((string) $title, $v['template']);

			/** @var Group $group */
			$group = $this->em->getRepository(Group::class)->find($groupId);
			$member->addGroup($group);

			foreach ($v['fields'] ?? [] as $kField => $vField) {
				$field = new Field($kField, $event->googleReview->{$vField}, $v['lang']);
				$this->em->persist($field);

				$member->fields->add($field);
			}

			$this->em->persist($member);
			$this->em->flush();

			$cache       = new Cache($this->cacheStorage, Groups::CACHE_NAMESPACE);
			$memberCache = new Cache($this->cacheStorage, Members::CACHE_NAMESPACE);

			foreach ($this->langsService->getLangs(false) as $lang) {
				$cache->remove($v['module'] . '_' . md5((string) $groupId) . '_' . $lang->getTag());

				$memberCache->remove('dynamicModuleMember/' . $lang->getTag() . '/' . $member->getId());
			}
			$cache->clean([Cache::TAGS => [Groups::CACHE_NAMESPACE]]);

			$cache = new Cache($this->cacheStorage, Members::CACHE_NAMESPACE);
			$cache->clean([Cache::TAGS => Members::CACHE_NAMESPACE]);
		}
	}
}
