<?php declare(strict_types = 1);

namespace DynamicModule\Model\TemplateTextType;

use Contributte\Translation\Translator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Event\TTPreload;
use Core\Model\TemplateReader\Providers\ITemplateTextType;
use Core\Model\TemplateReader\Providers\TemplateTextType;
use Doctrine\DBAL\Exception;
use DynamicModule\FrontModule\Model\Dao\Member;
use DynamicModule\FrontModule\Model\Repository\Members;
use Nette\InvalidArgumentException;
use Nette\Utils\Arrays;

class MembersTemplateText extends TemplateTextType implements ITemplateTextType
{
	/** @var string */
	protected                 $type = 'members';
	protected Members         $members;
	protected Translator      $translator;
	protected EventDispatcher $eventDispatcher;

	public function __construct(
		Members         $members,
		Translator      $translator,
		EventDispatcher $eventDispatcher
	)
	{
		$this->members         = $members;
		$this->translator      = $translator;
		$this->eventDispatcher = $eventDispatcher;
	}

	/**
	 * @return void
	 * @throws Exception
	 */
	public function loadContainer($formContainer, $params = [], $texts = [])
	{
		$membersAssoc = ['' => null];

		if (!isset($params['module'])) {
			throw new InvalidArgumentException('Parameter \'module\' must by specified');
		}

		$members = $this->members->getByModule($params['module'], false);
		foreach ($members as $member) {
			if (!isset($membersAssoc[$member->getId()]) || $member->getLang() === $this->translator->getLocale()) {
				$membersAssoc[$member->getId()] = $member->getTitle();
			}
		}

		if (isset($params['multiple']) && $params['multiple'] == true) {
			unset($membersAssoc[array_key_first($membersAssoc)]);

			if (isset($params['sortable']) && $params['sortable'] == true) {
				$formContainer->addSortableCheckboxList($this->getName(), $this->getTitle(), $membersAssoc);
			} else {
				$formContainer->addCheckboxList($this->getName(), $this->getTitle(), $membersAssoc);
			}
		} else {
			$formContainer->addSelect($this->getName(), $this->getTitle(), $membersAssoc);
		}
	}

	/**
	 * @param array $params
	 *
	 * @return Member[]|Member|null
	 * @throws Exception
	 */
	public function render($params)
	{
		$default = $this->getDefault();
		if (!$default) {
			return [];
		}

		if (isset($params['multiple']) && $params['multiple'] == true) {
			$result = $this->members->getByIds(is_array($default) ? $default : explode(',', $default));
			$fields = array_map(static fn($res) => $res->getKeyValueData(), $result);

			$this->eventDispatcher->dispatch(new TTPreload($fields, $result, $params, 'members'), 'core.ttPreload');

			if (isset($params['limit'])) {
				return array_slice($result, 0, (int) $params['limit']);
			}

			return $result;
		}

		$memberId = is_array($default) ? ((int) Arrays::first($default)) : ((int) $default);
		/** @var Member|null $member */
		$member   = $this->members->getByIds([$memberId])[$memberId];

		if ($member) {
			$fields = $member->getKeyValueData();
			$this->eventDispatcher->dispatch(new TTPreload($fields, [$member], $params, 'members'), 'core.ttPreload');
		}

		return $member;
	}
}
