<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components;

use Core\FrontModule\Model\SeoContainer;
use Core\Model\Entities\Seo;
use Core\Model\TemplateReader\TemplateReader;
use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use DynamicModule\AdminModule\Model\Events\MemberEvent;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Group;
use DynamicModule\Model\Entities\Member;
use DynamicModule\Model\Entities\MemberText;
use DynamicModule\Model\Helper;
use DynamicModule\Model\Repository\Groups;
use DynamicModule\Model\Repository\Members;
use Exception;
use Nette\Caching\Cache;
use Nette\ComponentModel\IComponent;
use Nette\Utils\ArrayHash;
use Nette\Utils\Validators;
use Tracy\Debugger;
use function str_contains;
use function str_ends_with;

/**
 * @property Member|null $entity
 */
class MemberForm extends TemplateReaderControl
{
	/** @var int|null @persistent */
	public ?int       $memberId = null;
	protected ?Member $member   = null;

	public function __construct(
		protected string       $moduleKey,
		protected Members      $members,
		protected Groups       $groups,
		protected SeoContainer $seoContainer,
	)
	{
	}

	/**
	 * TODO pokud bylo nějaké tt typu image, tak pri ulozeni
	 * formu a nacteni jine polozky zustaly v tt image stejne data.
	 *
	 * @param IComponent $presenter
	 */
	protected function attached($presenter): void
	{
		if ($this->memberId) {
			$this->entity = $this->members->get($this->memberId);
			$isGet        = $this->httpRequest->isMethod('GET');
			$q            = $this->httpRequest->getQuery('do');
			$postDo       = $this->httpRequest->getPost('_do');
			if (
				(!$isGet && $postDo && str_contains((string) $postDo, 'uploadFromServerForm'))
				|| ($isGet && $q && str_ends_with((string) $q, 'showGallery'))
				|| (!$isGet && $q && str_ends_with((string) $q, 'updateImage'))
				|| (!$isGet && $q && str_contains((string) $q, 'galleryGallery'))
				|| (!$isGet && $q && (str_ends_with((string) $q, 'Gallery-onEmpty')
						|| str_ends_with((string) $q, 'Gallery-upload')
						|| str_ends_with((string) $q, 'Gallery-removeImage')))
			) {
				$this->templateReader->loadTemplateComponents(
					$this['form']->getComponent('component'),
					$this->entity->template,
					$this->entity ? [$this->entity] : [],
				);
			}
		}
		parent::attached($presenter);
	}

	public function render(): void
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$this->templateReader->setTemplatesDir(
			sprintf('%s/Front/default/%s/TemplateMember', TEMPLATES_DIR, ucfirst($this->moduleKey)),
		);
		$this->templateReader->setTranslateKey(sprintf('%s.templateMember', $this->moduleKey));

		$templates = $this->templateReader->getTemplates();

		$groups = [];
		foreach ($this->groups->getQueryBuilderByModule($this->moduleKey)
			         ->orderBy('g.lft')
			         ->getQuery()->getResult() as $g) {
			$groups[$g->getId()] = trim((string) (str_repeat('---', $g->getLevel() - 1) . ' ' . $g->title));
		}

		$form = $this->createForm();
		$form->setAjax();

		$form->addText('title', 'default.title')
			->setRequired()
			->setMaxLength(255);
		$form->addBool('isPublished', 'default.isPublished')
			->setDefaultValue(1);
		if (DynamicModuleConfig::load('multiLangPublication')) {
			$form->getComponent('isPublished')->setIsMultilanguage();
		}
		$form->addCheckboxList('group', 'default.group', $groups)
			->setTranslator(null)
			->setRequired();
		$form->addSelect('template', $this->t('default.templates'), $templates)
			->setPrompt($this->t('default.choose'))
			->setRequired();
		$form->addComponent(new BaseContainer, 'component');
		$form->addHidden('id', $this->member ? $this->member->getId() : null);

		if (DynamicModuleConfig::load('member.allowSeo')) {
			$form->addComponent($this->seoContainer->getContainer(true), 'seo');
		}

		$form->addSaveCancelControl()
			->closeModalOnCancel();

		$form->onValidate[] = function() {
			$this->redrawControl('formErrors');
		};
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var MemberText[] $texts */
			$texts = [];

			if (!Validators::isNone($values->id)) {
				$this->member = $this->members->get((int) $values->id);
			}

			if (!$this->member) {
				$this->member = new Member($values->title, $values->template);
			} else {
				$texts = $this->member->texts->toArray();
			}

			$defaultGroupIds = array_map(static fn(Group $group): int => $group->getId(), $this->member->getGroups());

			// add to group
			foreach (array_diff($values->group, $defaultGroupIds) as $groupId) {
				$group = $this->groups->get((int) $groupId);

				if ($group) {
					$this->member->addGroup($group);
				}
			}

			// remove from group
			foreach (array_diff($defaultGroupIds, $values->group) as $groupId) {
				$group = $this->groups->get((int) $groupId);

				if ($group) {
					$this->member->removeGroup($group);
				}
			}

			$this->member->title    = $values->title;
			$this->member->template = $values->template;

			if (DynamicModuleConfig::load('multiLangPublication')) {
				foreach (array_keys($this->langsService->getLangs(false)) as $l) {
					if (!isset($langValues[$l]['isPublished'])) {
						$langValues[$l]['isPublished'] = 0;
					}
				}

				foreach ($langValues as $l => $v) {
					if (!isset($texts[$l])) {
						$texts[$l] = new MemberText($this->member, $l);
					}

					$texts[$l]->isPublished = (int) $v['isPublished'];

					$this->em->persist($texts[$l]);
				}
				$this->member->setEntityText($texts);
			} else {
				$this->member->isPublished = $values->isPublished;
			}

			// Nacteni komponent ze sablony
			$this->templateReader->loadTemplateComponents(
				$this['form']->getComponent('component'),
				$this->entity->template ?: $values->template,
				$this->entity ? [$this->entity] : [],
			);

			$data = Helper::transformTemplateArrayData(
				$form->getHttpData()['component'] ?: [],
				$form->getComponents(true),
			);
			$this->member->setTexts($data);

			if (DynamicModuleConfig::load('member.allowSeo')) {
				foreach ($langValues as $lang => $vals) {
					if (!isset($texts[$lang])) {
						$texts[$lang] = new MemberText($this->member, $lang);
					}

					$seo              = $texts[$lang]->seo ?: new Seo();
					$seo->title       = $vals['seo']['title'];
					$seo->description = $vals['seo']['description'];
					$this->em->persist($seo);
					$texts[$lang]->seo = $seo;
					$this->em->persist($texts[$lang]);
				}

				$this->member->setEntityText($texts);
			}

			$this->em->persist($this->member);
			$this->em->flush();

			$this->eventDispatcher->dispatch(
				new MemberEvent($this->member),
				'dynamicModule.admin.' . $this->member->groups->first()->group->moduleKey . '.memberChanged',
			);

			$form->addCustomData('memberId', $this->member->getId());
			$this->presenter->flashMessageSuccess('default.save');

			$cache  = new Cache(
				$this->cacheStorage, \DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE
			);
			$groups = array_merge($defaultGroupIds, $values->group);
			$groups = array_unique($groups);
			foreach ($this->langsService->getLangs(false) as $lang) {
				foreach ($groups as $groupId) {
					$cache->remove(md5((string) $groupId) . '_' . $lang->getTag());
				}
			}
			$cache->clean([Cache::Tags => [\DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE]]);

			$cache = new Cache($this->cacheStorage, Members::CACHE_NAMESPACE);
			$cache->clean([Cache::Tags => Members::CACHE_NAMESPACE]);
		} catch (Exception $e) {
			Debugger::log($e);
			$form->addError($e->getMessage());
			$this->redrawControl('formErrors');
		}
	}

	public function setMember(Member $member): void
	{
		$this->memberId = $member->getId();
		$this->member   = $member;

		$form = $this['form'];
		$form->setSubmittedBy(null);
		$form->setDefaults([
			'title'       => $member->title,
			'isPublished' => $member->isPublished,
			'template'    => $member->template,
			'id'          => $member->getId(),
			'group'       => array_map(static fn(Group $group): int => $group->getId(), $member->getGroups()),
		]);
		$form->getComponent('id')->setValue($member->getId());

		if (DynamicModuleConfig::load('multiLangPublication')) {
			$d = [];
			foreach ($this->member->getEntityTexts() as $l => $text) {
				$d['isPublished'][$l] = $text->isPublished;
			}

			if ($d) {
				$form->setDefaults($d);
			}
		}

		if (DynamicModuleConfig::load('member.allowSeo')) {
			$d = [];
			foreach ($member->getEntityTexts() as $lang => $text) {
				if (!$text->seo) {
					continue;
				}

				$d['seo']['title'][$lang]       = $text->seo->title;
				$d['seo']['description'][$lang] = $text->seo->description;
			}
			$form->setDefaults($d);
		}

		if ($this->member->template && array_key_exists(
				$this->member->template,
				$form->getComponent('template')
					->getItems(),
			)) {
			$form->getComponent('template')->setDefaultValue($this->member->template);

			if ($this->httpRequest->getQuery('do') !== 'memberForm-loadInputs') {
				/** @var BaseContainer $container */
				$container            = $form['component'];
				TemplateReader::$mode = 'dynamicModule';
				$this->templateReader->loadTemplateComponents(
					$container,
					$this->httpRequest->getPost('template') ?: $this->member->template,
					[$this->member],
				);
				$this->templateReader->setDefaults($container, $this->member, $this->member->template);
			}
		}
	}

}
