<?php declare(strict_types = 1);

namespace DynamicModule\FrontModule\Model\Repository;

use Core\Model\Helpers\BaseFrontEntityService;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use DynamicModule\FrontModule\Model\Dao;
use DynamicModule\FrontModule\Model\GroupQuery;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Group;
use Nette\Caching\Cache;

class Groups extends BaseFrontEntityService
{
	final public const CACHE_NAMESPACE = 'dynamicModuleGroups';

	/** @var string */
	protected $entityClass = Group::class;

	public function __construct(protected string $moduleKey, protected Fields $fields, protected Members $members)
	{
	}

	public function getCache(): Cache
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	public function getByIds(array $ids, ?string $lang = null): array
	{
		$lang    = $lang ?: $this->translator->getLocale();
		$process = function() use ($ids, $lang) {
			$repository = $this->getEr();

			$queryChild = new GroupQuery($this->moduleKey);
			if (!empty($ids)) {
				$queryChild->filterIds($ids);
			}
			$childQb = $queryChild->getQueryBuilder($repository);

			$queryRoot = new GroupQuery($this->moduleKey);
			if (!empty($ids)) {
				$queryRoot->filterIds($ids);
			}
			$rootQb = $queryRoot->getQueryBuilder($repository);

			if (DynamicModuleConfig::load('multiLangPublication')) {
				$childQb->join('node.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
				$childQb->setParameter('lang', $lang);
				$rootQb->join('node.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
				$rootQb->setParameter('lang', $lang);
			}

			if (!empty($ids))
				$rootQb->andWhere(
					'node.id IN (:ids) AND (node.lvl = 1 OR (node.lvl > 1 AND node.parent NOT IN (:ids2)))',
				)
					->setParameter('ids', $ids)
					->setParameter('ids2', $ids);

			return $this->buildTreeList($rootQb, $childQb, $lang);
		};

		if (DynamicModuleConfig::load('front.enableCacheGetByIds', true)) {
			$key = $this->moduleKey . '_' . md5(implode('_', $ids)) . '_' . $lang;

			return $this->getCache()->load($key, function(&$dep) use ($process) {
				$dep = [
					Cache::Tags       => [self::CACHE_NAMESPACE],
					Cache::Expire => '1 week',
				];

				return $process();
			});
		}

		return $process();
	}

	protected function buildTreeList(QueryBuilder $rootQb, QueryBuilder $childQb, ?string $lang = null): array
	{
		$childQb->andWhere('node.lvl > 0')
			->orderBy('node.lft'); // for children

		$tree     = [];
		$children = [];
		$groupIds = [];

		// prepare children
		$tmpChild = [];
		foreach ($childQb->getQuery()->getArrayResult() as $row) {
			$groupIds[] = $row[0]['id'];
			$tmpChild[] = $row;
		}

		// prepare root
		$tmpRoot = [];
		foreach ($rootQb->getQuery()->getArrayResult() as $row) {
			$groupIds[] = $row[0]['id'];
			$tmpRoot[]  = $row;
		}

		$fields  = $this->fields->getByGroups($groupIds, $lang);
		$members = $this->members->getByGroups($groupIds, $lang);

		// get all children
		foreach ($tmpChild as $row) {
			$parentId  = $row[0]['parent'] = $row['parent'];
			$moduleKey = $row[0]['moduleKey'];
			$id        = $row[0]['id'];

			$group                                = $this->fillDao($row[0]);
			$group->module                        = $moduleKey;
			$group->fields                        = $fields[$group->getId()] ?? [];
			$group->members                       = $members[$group->getId()] ?? [];
			$children[$moduleKey][$parentId][$id] = $group;

			foreach ($group->members as &$member) {
				$member->module = $moduleKey;
			}

			unset($member);
		}

		// get all roots and create tree
		foreach ($tmpRoot as $row) {
			$group     = $row[0];
			$moduleKey = $group['moduleKey'];

			$group['parent'] = null;
			$dao             = $this->fillDao($group);
			$dao->module     = $moduleKey;
			$dao->fields     = $fields[$dao->getId()] ?? [];
			$dao->members    = $members[$dao->getId()] ?? [];

			$children[$moduleKey][0][] = $dao;
			$tree[$moduleKey]          = $this->createTree($children[$moduleKey], $children[$moduleKey][0]);

			foreach ($dao->members as &$member) {
				$member->module = $moduleKey;
			}

			unset($member);
		}

		$result = [];

		if (isset($moduleKey)) {
			$result = $tree[$moduleKey] ?? [];
		}

		// set parent for second level children
		foreach ($result as $item) {
			foreach ($item->children as $ch) {
				$ch->parent = $item;
			}
		}

		return $result;
	}

	protected function fillDao(array $fields): Dao\Group
	{
		$group          = new Dao\Group((int) $fields['id'], $fields['title'], $fields['template']);
		$group->lvl     = (int) $fields['lvl'];
		$group->parent  = $fields['parent'];
		$group->fields  = [];
		$group->members = [];

		if ($group->lvl === 1) {
			$group->parent = 0;
		}

		return $group;
	}

	/**
	 * @param Dao\Group[] $children
	 * @param Dao\Group[] $parents
	 *
	 * @return Dao\Group[]
	 */
	protected function createTree(array &$children, array $parents): array
	{
		$tree = [];
		foreach ($parents as $k => $l) {
			if (isset($children[$l->id])) {
				if ($l->lvl >= 1) {
					foreach ($children[$l->id] as $kk => $v) {
						$children[$l->id][$kk]->parent = &$parents[$k];
					}
				}
				$l->children = $this->createTree($children, $children[$l->id]);
			}

			$tree[$l->id] = $l;
		}

		return $tree;
	}

}
