<?php declare(strict_types = 1);

namespace DynamicModule\FrontModule\Model\Repository;

use Contributte\Translation\Translator;
use Core\FrontModule\Model\Dao\Seo;
use Core\Model\Helpers\BaseEntityService;
use Doctrine\DBAL\Exception;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use DynamicModule\FrontModule\Model\Dao;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\GroupMember;
use DynamicModule\Model\Entities\Member;
use Nette\Caching\Cache;
use function str_contains;

class Members extends BaseEntityService
{
	/** @var string */
	protected       $entityClass = Member::class;
	protected array $cMembers    = [];

	public function __construct(protected Fields $fields, protected Translator $translator)
	{
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, \DynamicModule\Model\Repository\Members::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	protected function joinSeo(QueryBuilder $qb): void
	{
		if (DynamicModuleConfig::load('member.allowSeo')) {
			if (!in_array('txt', $qb->getAllAliases(), true)) {
				$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang')
					->setParameter('lang', $this->translator->getLocale());
			}
			$qb->leftJoin('txt.seo', 'seo')
				->addSelect('seo.title as seoTitle, seo.description as seoDescription');
		}
	}

	protected function fillSeoDao(array $data): ?Seo
	{
		if (DynamicModuleConfig::load('member.allowSeo')) {
			$seo              = new Seo();
			$seo->title       = $data['seoTitle'] ?: null;
			$seo->description = $data['seoDescription'] ?: null;

			if ($seo->title && !str_contains($seo->title, '$siteName')) {
				$seo->title = trim($seo->title . ' $separator $siteName');
			}

			return $seo;
		}

		return null;
	}

	public function getById(int $memberId): ?Dao\Member
	{
		$qb = $this->getEr()->createQueryBuilder('m');
		$qb->select('m.id, m.lang, m.template, m.title, g.moduleKey')
			->andWhere('m.id = :memberId')
			->join('m.groups', 'gm')
			->join('gm.group', 'g')
			->setParameter('memberId', $memberId);

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$qb->andWhere('m.isPublished = 1');
		} else {
			$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
			$qb->setParameter('lang', $this->translator->getLocale());
		}

		$this->joinSeo($qb);

		$row = $qb->getQuery()->getOneOrNullResult();

		if (!$row) {
			return null;
		}

		$member         = new Dao\Member($row['id'], $row['title'], $row['template']);
		$member->module = $row['moduleKey'];
		$member->lang   = $row['lang'] ?? null;
		$member->fields = [];

		$seo = $this->fillSeoDao($row);
		if ($seo) {
			$member->seo = $seo;
		}

		$fields = $this->fields->getByMembers([$row['id']]);
		if (count($fields) > 0) {
			$member->fields = $fields[array_key_first($fields)];
		}

		return $member;
	}

	/**
	 * @param int[] $ids
	 *
	 * @return Dao\Member[]
	 * @throws Exception
	 * @throws Exception
	 */
	public function getByIds(array $ids): array
	{
		$result = [];
		if (!$ids) {
			return [];
		}

		foreach ($ids as $id) {
			$keys[] = 'dynamicModuleMember-' . $this->translator->getLocale() . ':' . $id;
		}

		$whereIds = [];
		foreach ($this->getCache()->bulkLoad($keys) as $key => $member) {
			$tmp = explode(':', $key);

			if ($member) {
				$result[$tmp[1]] = $member;
			} else {
				$whereIds[] = $tmp[1];
			}
		}

		if (!empty($whereIds)) {
			$qb = $this->getEr()->createQueryBuilder('m');
			$qb->select('m.id, m.lang, m.template, m.title, g.moduleKey')
				->andWhere('m.id IN(:memberIds)')
				->join('m.groups', 'gm')
				->join('gm.group', 'g')
				->setParameter('memberIds', $ids);

			if (!DynamicModuleConfig::load('multiLangPublication')) {
				$qb->andWhere('m.isPublished = 1');
			} else {
				$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
				$qb->setParameter('lang', $this->translator->getLocale());
			}

			$this->joinSeo($qb);

			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$member         = new Dao\Member($row['id'], $row['title'], $row['template']);
				$member->module = $row['moduleKey'];
				$member->lang   = $row['lang'] ?? null;
				$member->fields = [];

				$seo = $this->fillSeoDao($row);
				if ($seo)
					$member->seo = $seo;

				$fields = $this->fields->getByMembers([$row['id']]);
				if (count($fields) > 0) {
					$member->fields = $fields[array_key_first($fields)];
				}
				$result[$member->getId()] = $member;
				$this->getCache()->save('dynamicModuleMember:' . $member->getId(), $member, [
					Cache::Tags       => [\DynamicModule\Model\Repository\Members::CACHE_NAMESPACE],
					Cache::Expire => '1 week',
				]);
			}
		}

		return $result;
	}

	/**
	 * @return Dao\Member[]
	 * @throws Exception
	 */
	public function getByGroup(int $groupId): array
	{
		$qb = $this->getEr()->createQueryBuilder('m');
		$qb->select('m.id, m.lang, m.template, m.title, g.moduleKey')
			->join('m.groups', 'gm')
			->join('gm.group', 'g')
			->andWhere('g.id = :groupId')
			->addOrderBy('gm.position')
			->setParameter('groupId', $groupId);

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$qb->andWhere('m.isPublished = 1');
		} else {
			$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
			$qb->setParameter('lang', $this->translator->getLocale());
		}

		$this->joinSeo($qb);

		$result = [];
		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$member         = new Dao\Member($row['id'], $row['title'], $row['template']);
			$member->lang   = $row['lang'] ?? null;
			$member->module = $row['moduleKey'];
			$member->fields = [];

			$seo = $this->fillSeoDao($row);
			if ($seo) {
				$member->seo = $seo;
			}

			$result[$row['id']] = $member;
		}

		foreach ($this->fields->getByMembers(array_keys($result)) as $k => $v) {
			$result[$k]->fields = $v;
		}

		return $result;
	}

	/**
	 * @return Dao\Member[]
	 * @throws Exception
	 */
	public function getByModule(string $module, bool $onlyPublished = true): array
	{
		$qb = $this->getEr()->createQueryBuilder('m');
		$qb->select('m.id, m.lang, m.template, m.title')
			->join('m.groups', 'gm')
			->join('gm.group', 'g')
			->andWhere('g.moduleKey = :module')
			->setParameter('module', $module)
			->addOrderBy('m.title');

		if ($onlyPublished) {
			if (!DynamicModuleConfig::load('multiLangPublication')) {
				$qb->andWhere($qb->expr()->eq('m.isPublished', (int) $onlyPublished));
			} else {
				$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
				$qb->setParameter('lang', $this->translator->getLocale());
			}
		}

		$this->joinSeo($qb);

		$result = [];
		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$member         = new Dao\Member($row['id'], $row['title'], $row['template']);
			$member->module = $module;
			$member->lang   = $row['lang'] ?? null;
			$member->fields = [];

			$seo = $this->fillSeoDao($row);
			if ($seo) {
				$member->seo = $seo;
			}

			$result[$row['id']] = $member;
		}

		foreach ($this->fields->getByMembers(array_keys($result)) as $k => $v) {
			$result[$k]->fields = $v;
		}

		return $result;
	}

	public function getByGroups(array $groupIds, ?string $lang = null): array
	{
		$result = [];
		$lang   = $lang ?: $this->translator->getLocale();

		$structured     = [];
		$memberIds      = [];
		$membersForLoad = [
			$lang => [],
		];

		if (!isset($this->cMembers[$lang])) {
			$this->cMembers[$lang] = [];
		}

		// vyhledani id clenu pro skupiny
		foreach ($this->em->getRepository(GroupMember::class)->createQueryBuilder('gm')
			         ->select('IDENTITY(gm.group) as group, IDENTITY(gm.member) as member')
			         ->orderBy('gm.position')
			         ->andWhere('gm.group IN (:groups)')
			         ->setParameter('groups', $groupIds)
			         ->getQuery()->getArrayResult() as $row) {
			$structured[$row['group']][] = $row['member'];
			$memberIds[]                 = $row['member'];

			if (!isset($this->cMembers[$lang][$row['member']])) {
				$membersForLoad[$lang][] = $row['member'];
			}
		}

		// nacteni zbyvajicich clenu
		if (!empty($membersForLoad[$lang])) {
			$qb = $this->getEr()->createQueryBuilder('m');
			$qb->select('m.id, m.lang, m.template, m.title, GROUP_CONCAT(IDENTITY(g.group)) as groups')
				->andWhere('m.id IN (:ids)')
				->leftJoin('m.groups', 'g')
				->groupBy('m.id')
				->setParameter('ids', array_unique($membersForLoad[$lang]));

			if (!DynamicModuleConfig::load('multiLangPublication')) {
				$qb->andWhere('m.isPublished = 1');
			} else {
				$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
				$qb->setParameter('lang', $lang);
			}

			$this->joinSeo($qb);

			$loaded = [];
			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$member         = new Dao\Member($row['id'], $row['title'], $row['template']);
				$member->lang   = $row['lang'] ?? null;
				$member->fields = [];
				$member->setGroupIds(explode(',', (string) $row['groups']));

				$seo = $this->fillSeoDao($row);
				if ($seo) {
					$member->seo = $seo;
				}

				$loaded[$row['id']] = $member;
			}

			foreach ($this->fields->getByMembers(array_keys($loaded)) as $k => $v) {
				$loaded[$k]->fields        = $v;
				$this->cMembers[$lang][$k] = $loaded[$k];
			}
		}

		// slozeni vysledku
		foreach ($structured as $groupId => $memberIds) {
			foreach ($memberIds as $memberId) {
				if (isset($this->cMembers[$lang][$memberId])) {
					$result[$groupId][$memberId] = $this->cMembers[$lang][$memberId];
				}
			}
		}

		return $result;
	}
}
