<?php declare(strict_types = 1);

namespace DynamicModule\FrontModule\Presenters;

use Core\Components\Navigation\DaoNavigationItem;
use DynamicModule\FrontModule\Model\Repository\IGroupsFactory;
use DynamicModule\FrontModule\Model\Repository\Members;
use Pages\FrontModule\Model\TemplatePages;
use Pages\Model\Helpers\TemplatePagesHelper;
use Pages\Model\Paths;

class DefaultPresenter extends BasePresenter
{
	public function __construct(
		protected TemplatePages       $templatePagesService,
		protected Paths               $pathsService,
		protected TemplatePagesHelper $pagesHelper,
		protected IGroupsFactory      $groupsFrontRepoFactory,
		protected Members             $members,
	)
	{
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionGroups(): void
	{
		$activeNav      = $this->getActiveNavigation();
		$templatePageId = (int) $activeNav->componentParams['templatePage'];
		$page           = $templatePageId ? $this->templatePagesService->get($templatePageId) : null;

		if (!$page) {
			$this->error('pages.page.notFound');
		}

		$this->template->pageClass .= " template-page tp-{$page->getId()} dynamic-module dm-groups";

		$groupsRepo             = $this->groupsFrontRepoFactory->create($activeNav->componentParams['module']);
		$groups                 = $groupsRepo->getByIds([]);
		$this->template->groups = $groups;

		$texts = $page->getTexts();
		$texts = $this->pagesHelper->checkTexts($texts);

		$this->template->templateTextValues = $texts;
		$this->template->setFile($this->pathsService->getTemplatePagesFile($page->getTemplate()));
	}

	public function actionGroup(string $id): void
	{
		$activeNav      = $this->getActiveNavigation();
		$templatePageId = (int) ($activeNav->componentParams['templatePageDetail'] ?: $activeNav->componentParams['templatePage']);
		$page           = $this->templatePagesService->get($templatePageId);

		if (!$page) {
			$this->error('pages.page.notFound');
		}

		$this->template->pageClass .= " template-page tp-{$page->getId()} dynamic-module dm-{$id}";

		$groupsRepo             = $this->groupsFrontRepoFactory->create($activeNav->componentParams['module']);
		$groups                 = $groupsRepo->getByIds($activeNav->componentParams['baseGroup'] ?? []);
		$this->template->groups = $groups;

		$group = $groups[$id] ?? null;
		if (!$group && !empty($groups)) {
			$group = array_values($groups)[0];
		}

		$this->template->group = $group;

		$texts = $page->getTexts();
		$texts = $this->pagesHelper->checkTexts($texts);

		$this['meta']->setMeta(
			'canonical',
			$this->getHttpRequest()
				->getUrl()
				->getBaseUrl() . ltrim($this->getActiveNavigation()->link, '/'),
		);

		if ($group) {
			$this->breadcrumbsAfter[] = new DaoNavigationItem([
				'title' => $group->getFieldValue('title') ?: $group->title,
				'link'  => $this->getHttpRequest()->getUrl()->getAbsoluteUrl(),
			]);
		}

		$this->template->templateTextValues = $texts;
		$this->template->setFile($this->pathsService->getTemplatePagesFile($page->getTemplate()));
	}

	public function actionMember(int $id): void
	{
		$activeNav = $this->getActiveNavigation();
		if (!$activeNav->componentParams['allowMemberDetail'])
			$this->error('pages.page.notFound');

		$groupsRepo             = $this->groupsFrontRepoFactory->create($activeNav->componentParams['module']);
		$groups                 = $groupsRepo->getByIds($activeNav->componentParams['baseGroup'] ?? []);
		$this->template->groups = $groups;
		$member                 = null;

		foreach ($groups as $group) {
			$member = $group->findMember($id);
			if ($member) {
				$this->template->member = $member;
				$this->template->group  = $group;

				$groupLink = $this->link('group', ['id' => $group->getId()]);
				if ($groupLink && $groupLink !== '#') {
					$this->breadcrumbsAfter[] = new DaoNavigationItem([
						'title' => $group->getFieldValue('title') ?: $group->getTitle(),
						'link'  => $groupLink,
					]);
				}

				$this->template->templateTextValues = $member->getKeyValueData();
				break;
			}
		}

		if (!$member)
			$this->error('pages.page.notFound');

		if ($member->seo) {
			$this->seoDataFromEntity['title']       = $member->seo->title;
			$this->seoDataFromEntity['description'] = $member->seo->description;
		}

		if (!$this->seoDataFromEntity['title'])
			$this->setTitle($activeNav->title . ' - ' . ($member->getFieldValue('title') ?: $member->getTitle()));

		$this->breadcrumbsAfter[] = new DaoNavigationItem([
			'title' => $member->getFieldValue('title') ?: $member->title,
			'link'  => $this->getHttpRequest()->getUrl()->getAbsoluteUrl(),
		]);

		$templatesDir = sprintf(
			'%s/Front/default/%s/TemplateMember',
			TEMPLATES_DIR,
			ucfirst((string) $activeNav->componentParams['module']),
		);

		if ($activeNav->componentParams['templatePageMemberDetail']) {
			$page = $this->templatePagesService->get((int) $activeNav->componentParams['templatePageMemberDetail']);

			if (!$page) {
				$this->error('pages.page.notFound');
			}

			$this->template->pageClass .= " template-page tp-{$page->getId()} dynamic-module dm-{$id}";

			$texts = $page->getTexts();
			$texts = $this->pagesHelper->checkTexts($texts);

			$this->template->templateTextValues = $texts;
			$this->template->setFile($this->pathsService->getTemplatePagesFile($page->getTemplate()));
		} else {
			$this->template->setFile($templatesDir . DS . $this->template->member->getTemplate());
		}
	}
}
