<?php declare(strict_types = 1);

namespace DynamicModule\Model\Console;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use Doctrine\ORM\Tools\SchemaTool;
use DynamicModule\Model\DynamicModuleConfig;
use Exception;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class FillEntityText extends AbstractCommand
{
	/** @var EntityManagerDecorator @inject */
	public EntityManagerDecorator $em;

	/** @var Translator @inject */
	public Translator $translator;

	protected function configure(): void
	{
		$this->setName('dynamic-module:fill-entity-text')->setDescription('Add translation text which missed');
	}

	protected function executeSchemaCommand(
		InputInterface  $input,
		OutputInterface $output,
		SchemaTool      $schemaTool,
		array           $metadatas,
		SymfonyStyle    $ui,
	): int
	{
		$output->writeln('WORKING...');

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$output->writeln('You must enable multiLangPublication in configuration');

			return Command::FAILURE;
		}

		$connection = $this->em->getConnection();
		try {
			$connection->transactional(function() use ($connection, $output) {
				$langs         = $this->translator->getLocalesWhitelist();
				$defaultLocale = $this->translator->getDefaultLocale();
				$groups        = $connection->fetchAllAssociative("SELECT id FROM dynamicmodule__group");
				$groupTexts    = $connection->fetchAllAssociative("SELECT group_id id, lang FROM dynamicmodule__group_text");
				$members       = $connection->fetchAllAssociative("SELECT id FROM dynamicmodule__member");
				$memberTexts   = $connection->fetchAllAssociative(
					"SELECT member_id id, lang, seo_id FROM dynamicmodule__member_text",
				);
				$isMissingText = false;

				foreach ($groups as $g) {
					$groupId = $g['id'];
					$texts   = [];
					foreach ($groupTexts as $gt) {
						if ($groupId === $gt['id']) {
							$texts[$gt['lang']] = $gt;
						}
					}

					foreach ($langs as $lang) {
						if (!array_key_exists($lang, $texts)) {
							$isMissingText = true;

							$connection->executeQuery("INSERT INTO dynamicmodule__group_text (lang, group_id, is_published) VALUES (?, ?, ?)", [
								$lang,
								$groupId,
								1,
							]);
						}
					}
				}

				foreach ($members as $m) {
					$memberId = $m['id'];
					$texts    = [];
					foreach ($memberTexts as $mt) {
						if ($memberId === $mt['id']) {
							$texts[$mt['lang']] = $mt;
						}
					}

					foreach ($langs as $lang) {
						if (!array_key_exists($lang, $texts)) {
							$isMissingText = true;

							$seoId = null;
							if (array_key_exists($defaultLocale, $texts) && $m['seo_id']) {
								$connection->prepare("INSERT INTO core__seo (title, description) VALUES (?, ?)")
									->execute([
										'', '',
									]);
								$seoId = $connection->lastInsertId();
							}

							$connection->prepare(
								"INSERT INTO dynamicmodule__member_text (lang, member_id, is_published, seo_id) VALUES (?, ?, ?, ?)",
							)
								->execute([
									$lang,
									$memberId,
									1,
									$seoId,
								]);
						}
					}
				}

				if (!$isMissingText) {
					$output->writeln('All langs already filled');
				}

				if ($isMissingText) {
					$output->writeln('COMPLETED');
				}
			});
		} catch (Exception) {
			$output->writeln('ERROR, PLEAS CHECK LOG');

			return Command::FAILURE;
		}

		return Command::SUCCESS;
	}

}
