<?php declare(strict_types = 1);

namespace DynamicModule\Model\Repository;

use Contributte\Translation\Translator;
use Core\Model\Entities\Repository\NestedTreeRepository;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\CoreHelper;
use Core\Model\Helpers\Traits\TPublish;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Group;
use Nette\Utils\Strings;
use Tracy\Helpers;

/**
 * @method Group|object|null getReference($id)
 * @method Group[] getAll()
 * @method Group|null get(int $id)
 * @method NestedTreeRepository getEr()
 */
class Groups extends BaseEntityService
{
	use TPublish;

	/** @var string */
	protected        $entityClass = Group::class;
	protected ?array $checkRoute  = null;

	public function __construct(protected Translator $translator)
	{
	}

	public function getQueryBuilderByModule(string $moduleKey): QueryBuilder
	{
		$qb = $this->getEr()->createQueryBuilder('g');
		$qb->andWhere('g.moduleKey = :module')
			->andWhere('g.lvl != 0')
			->setParameter('module', $moduleKey);

		return $qb;
	}

	public function checkRoute(string|int $groupId, string $module): ?array
	{
		$mKey = $groupId . '_' . $module;

		if ($this->checkRoute === null) {
			$this->checkRoute = [];

			$qb = $this->getEr()->createQueryBuilder('g');
			$qb->select('g.id, g.title, g.moduleKey');

			if (DynamicModuleConfig::load('multiLangPublication')) {
				$qb->join('g.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
				$qb->setParameter('lang', $this->translator->getLocale());
			}

			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$arr = [
					'id'    => $row['id'],
					'title' => $row['title'],
					'url'   => Strings::webalize($row['title']),
				];

				$this->checkRoute[$row['id'] . '_' . $row['moduleKey']]  = $arr;
				$this->checkRoute[$arr['url'] . '_' . $row['moduleKey']] = $arr;
			}
		}

		return $this->checkRoute[$mKey] ?? null;
	}

	public function getRootGroup(string $moduleKey): ?Group
	{
		return $this->getEr()->createQueryBuilder('g')
			->where('g.lvl = 0')
			->andWhere('g.title = :moduleKey')
			->setParameters([
				'moduleKey' => $moduleKey,
			])
			->setMaxResults(1)
			->getQuery()->getOneOrNullResult();
	}

	public function checkRoots(array $navigations): void
	{
		if (CoreHelper::isCompiled() || Helpers::isCli()) {
			return;
		}

		$keys = [];

		foreach ($navigations as $sections) {
			foreach ($sections as $key => $navigation) {
				if ($navigation['link'][0] === ':DynamicModule:Admin:Default:' && isset($navigation['link'][1]['id'])) {
					$keys[] = $navigation['link'][1]['id'];
				}
			}
		}

		foreach (array_unique($keys) as $module) {
			$this->checkRoot($module);
		}
	}

	public function checkRoot(string $moduleKey): void
	{
		$root = $this->getEr()->findOneBy(['title' => $moduleKey, 'lvl' => 0]);

		if ($root) {
			return;
		}

		$root           = new Group($moduleKey, $moduleKey);
		$root->template = '';
		$root->setParent(null);
		$this->em->persist($root);

		foreach ($this->getEr()->createQueryBuilder('g')
			         ->where('g.lvl = :lvl')
			         ->andWhere('g.moduleKey = :module')
			         ->setParameters([
				         'module' => $moduleKey,
				         'lvl'    => 0,
			         ])->orderBy('g.title')
			         ->getQuery()->getResult() as $row) {
			/** @var Group $row */
			$row->setParent($root);
			$this->em->persist($row);
		}

		$this->em->flush();
	}
}
