<?php declare(strict_types = 1);

namespace DynamicModule\Model\Repository;

use Contributte\Translation\Translator;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\Helpers\Traits\TPublish;
use Core\Model\Lang\Langs;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\GroupMember;
use DynamicModule\Model\Entities\Member;
use Nette\Caching\Cache;
use Throwable;

/**
 * @method Member[] getAll()
 * @method Member|null get(int $id)
 */
class Members extends BaseEntityService
{
	use TPublish;

	final public const CACHE_NAMESPACE = 'dynamicModuleMembers';

	/** @var string */
	protected $entityClass = Member::class;
	/** @var string[] */
	protected array $checkRoute = [];

	public function __construct(protected Translator $translator, protected Langs $langs)
	{
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	public function getQueryBuilderByModule(string $moduleKey): QueryBuilder
	{
		$qb = $this->getEr()->createQueryBuilder('m');
		$qb->join('m.groups', 'gm', Join::WITH, 'm.id = gm.member')
			->addSelect('gm')
			->join('gm.group', 'g')
			->andWhere('g.moduleKey = :module')
			->setParameter('module', $moduleKey);

		return $qb;
	}

	public function getQueryBuilderByGroup(int $groupId): QueryBuilder
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('gm')
			->from(GroupMember::class, 'gm')
			->join('gm.group', 'g')
			->where($qb->expr()->eq('g.id', $groupId))
			->addOrderBy('gm.position');

		return $qb;
	}

	/**
	 * @throws Throwable
	 */
	public function checkRoute(
		?array          $groups,
		string          $module,
		string|int|null $key = null,
		array           $moreFields = [],
	): ?array
	{
		if (!is_array($groups)) {
			$groups = [];
		}

		$mKey = $module . serialize($groups) . (!empty($moreFields) ? ':' . serialize(
					$moreFields,
				) : '') . '_' . $this->translator->getLocale();
		if (!isset($this->checkRoute[$mKey])) {
			$this->checkRoute[$mKey] = $this->getCache()->load($mKey, function(&$dep) use (
				$groups,
				$module,
				$moreFields,
			) {
				$dep = [
					Cache::Tags       => [self::CACHE_NAMESPACE, self::CACHE_NAMESPACE . 'checkRoute' . $module],
					Cache::Expire => '1 week',
				];

				$checkRoute = [];

				$rows = $this->getEr()->createQueryBuilder('m')
					->select('m.id, m.title');

				if (!empty($groups)) {
					$rows->innerJoin('m.groups', 'gm', Join::WITH, 'gm.group IN (:groups)')
						->setParameter('groups', $groups);
				} else {
					$rows->innerJoin('m.groups', 'gm');
				}

				$rows->innerJoin('gm.group', 'g', Join::WITH, 'g.moduleKey = :module')
					->setParameter('module', $module);

				if (DynamicModuleConfig::load('multiLangPublication')) {
					$rows->innerJoin('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
					$rows->setParameter('lang', $this->translator->getLocale());
				}

				foreach ($rows->getQuery()->getArrayResult() as $row) {
					$checkRoute[$row['id']] = [
						'id'    => $row['id'],
						'title' => $row['title'],
					];
				}

				if (!empty($checkRoute)) {
					$moreFields = array_merge($moreFields, ['title', 'alias', 'fullUrl', 'gallery']);

					foreach ($this->em->getConnection()->fetchAllAssociative(
						"SELECT mf.member_id, f.key, f.value FROM dynamicmodule__field f
    				INNER JOIN dynamicmodule__member_fields mf ON mf.field_id = f.id AND mf.member_id IN (" . implode(
							',',
							array_keys($checkRoute),
						) . ")
    				WHERE f.key IN ('" . implode(
							'\',\'',
							$moreFields,
						) . "') AND (f.lang = '{$this->translator->getLocale()}' OR f.lang IS NULL)",
					) as $row) {
						$checkRoute[$row['member_id']][$row['key']] = $row['value'];
					}

					foreach ($checkRoute as $id => $vals) {
						if ($vals['fullUrl'] ?? null) {
							$url = $vals['fullUrl'];
						} else {
							$url = $id . '-' . ($vals['alias'] ?? Strings::webalize($vals['title']));
						}

						$url = Strings::webalize($url);

						$checkRoute[$id]['url'] = $url;
						$checkRoute[$url]       = $checkRoute[$id];
					}
				}

				return $checkRoute;
			});
		}

		return $key ? ($this->checkRoute[$mKey][$key] ?? null) : $this->checkRoute[$mKey];
	}

	/**
	 * @param string|int $id
	 */
	public function remove($id): bool
	{
		$member = $this->get($id);
		if (!$member) {
			return false;
		}

		$groups = array_map(static fn($g) => (string) $g->getId(), $member->getGroups());
		$result = parent::remove($id);

		$cache = new Cache($this->cacheStorage, \DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE);
		foreach ($this->langs->getLangs(false) as $lang) {
			foreach ($groups as $groupId) {
				$cache->remove(md5($groupId) . '_' . $lang->getTag());
			}
		}
		$cache->clean([Cache::Tags => [\DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE]]);

		return $result;
	}

}
