<?php declare(strict_types = 1);

namespace DynamicModule\Model\TemplateTextType;

use Contributte\Latte\Exception\Logical\InvalidArgumentException;
use Contributte\Translation\Translator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Event\TTPreload;
use Core\Model\Helpers\Arrays;
use Core\Model\TemplateReader\Providers\ITemplateTextType;
use Core\Model\TemplateReader\Providers\TemplateTextType;
use Doctrine\DBAL\Exception;
use DynamicModule\FrontModule\Model\Dao\Field;
use DynamicModule\FrontModule\Model\Dao\Group;
use DynamicModule\FrontModule\Model\Dao\Member;
use DynamicModule\FrontModule\Model\Repository\IGroupsFactory;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Repository\Groups;
use DynamicModule\Model\Repository\Members;
use EshopCatalog\DI\EshopCatalogExtension;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use Nette\DI\Container;
use Nette\Utils\DateTime;
use Pages\Model\TemplateTextType as PageTemplateTextType;
use Throwable;

class GroupMembersTemplateText extends TemplateTextType implements ITemplateTextType
{
	/** @var string */
	protected $type = 'groupMembers';

	public function __construct(
		protected Groups          $groups,
		protected Members         $members,
		protected IGroupsFactory  $groupsFrontRepoFactory,
		protected EventDispatcher $eventDispatcher,
		protected Container       $diContainer,
		protected Translator      $translator,
	)
	{
	}

	/**
	 * @return void
	 * @throws Exception
	 */
	public function loadContainer($formContainer, $params = [], $texts = [])
	{
		if (isset($params['default'])) {
			return;
		}

		if (!isset($params['module'])) {
			throw new \Nette\InvalidArgumentException('Parameter \'module\' must by specified');
		}

		$qb = $this->groups->getQueryBuilderByModule($params['module'])
			->addOrderBy('g.id')
			->addOrderBy('g.title');

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$qb->andWhere('g.isPublished = 1');
		}

		$groups = [];
		foreach ($qb->getQuery()->getResult() as $g) {
			$groups[] = [
				'id'     => $g->getId(),
				'parent' => $g->getLevel() === 1 ? 0 : ($g->parent ? $g->parent->getId() : 0),
				'name'   => $g->title,
			];
		}

		$formContainer->addCheckboxNestedList($this->getName(), $this->getTitle(), $groups);
	}

	/**
	 * @param array $params
	 *
	 * @return array
	 * @throws Throwable
	 */
	public function render($params)
	{
		if (!isset($params['toVar'])) {
			throw new InvalidArgumentException('Need variable for output');
		}

		if (isset($params['default'])) {
			$this->default = explode(',', (string) $params['default']);
		}

		$moduleName = $params['module'] ?? null;

		if (!$moduleName) {
			throw new InvalidArgumentException('Module name is missing');
		}

		if (!is_iterable($this->getDefault()) || count($this->getDefault()) === 0) {
			return [];
		}

		$groups = $this->groupsFrontRepoFactory->create($moduleName);

		/** @var Group[] $result */
		$result = $groups->getByIds($this->getDefault());

		if (isset($params['replaceMissingInLangs'][$this->translator->getLocale()])) {
			$locale = $this->translator->getLocale();
			$this->translator->setLocale($params['replaceMissingInLangs'][$this->translator->getLocale()]);

			/** @var Group[] $result2 */
			$result2 = $groups->getByIds($this->getDefault(), $this->translator->getLocale());

			$this->translator->setLocale($locale);

			foreach ($result as $gk => $gv) {
				foreach ($gv->getMembers() as $mk => $mv) {
					foreach ($mv->getFields() as $fv) {
						/** @var Field $fv */
						if (!$fv->getValue() && isset($result2[$gk], $result2[$gk]->getMembers()[$mk])) {
							$fv->setValue($result2[$gk]->getMembers()[$mk]->getFieldValue($fv->getKey()));
						}
					}
				}

				foreach ($gv->getFields() as $fv) {
					if (!$fv->getValue() && isset($result2[$gk])) {
						$fv->setValue($result2[$gk]->getFieldValue($fv->getKey()));
					}
				}
			}
		}

		$members = [];
		if (isset($params['onlyMembers'])) {
			$loopGroups = static function(array $groups) use (&$members, &$loopGroups) {
				foreach ($groups as $group) {
					foreach ($group->getMembers() as $member) {
						$members[$member->getId()] = $member;
					}

					if ($group->getChildren()) {
						$loopGroups($group->getChildren());
					}
				}
			};

			$loopGroups($result);

			$result = $members;
		}

		if (isset($params['dateFromTo'])) {
			$now             = new DateTime;
			$checkDateFromTo = static function(Member $member) use ($now): bool {
				$from = PageTemplateTextType\DateTime::getDateTime($member->getFieldValue('dateFrom'));
				$to   = PageTemplateTextType\DateTime::getDateTime($member->getFieldValue('dateTo'));

				return (
					(!$from || $from->getTimestamp() < $now->getTimestamp())
					&& (!$to || $to->getTimestamp() >= $now->getTimestamp())
				);
			};

			foreach ($result as $resultK => $value) {
				if ($value instanceof Group) {
					foreach ($value->getMembers() as $k => $member) {
						if (!$checkDateFromTo($member)) {
							$value->removeMember($k);
						}
					}
				} else if ($value instanceof Member) {
					unset($result[$resultK]);
				}
			}
		}

		if (isset($params['shuffle']) && $result) {
			$result = Arrays::shuffleAssoc($result);
		}

		if (isset($params['limit'])) {
			$result = array_slice($result, 0, (int) $params['limit'], true);
		}

		if (class_exists(EshopCatalogExtension::class) && $this->diContainer->hasService(
				'eshopCatalog.front.productsFacade',
			)) {
			$products = [];

			if (class_exists(ProductsFacade::class)) {
				/** @var ProductsFacade $productsFacade */
				$productsFacade = $this->diContainer->getService('eshopCatalog.front.productsFacade');
				foreach ($result as $group) {
					if ($group instanceof Group) {
						foreach ($group->getMembers() as $member) {
							if ($member->getFieldValue('productId')) {
								$products[] = $member->getFieldValue('productId');
							}
						}
					} else if ($group instanceof Member) {
						if ($group->getFieldValue('productId')) {
							$products[] = $group->getFieldValue('productId');
						}
					}
				}

				if ($products) {
					$productsFacade->getProducts($products);
				}
			}
		}

		$fields = [];
		foreach ($result as $group) {
			if ($group instanceof Group) {
				foreach ($group->getMembers() as $member) {
					$fields[] = $member->getKeyValueData();
				}
			} else if ($group instanceof Member) {
				$fields[] = $group->getKeyValueData();
			}
		}
		if ($fields) {
			$this->eventDispatcher->dispatch(new TTPreload($fields, $result, $params, 'members'), 'core.ttPreload');
		}

		return $result;
	}

}
