<?php declare(strict_types = 1);

namespace Ecomail\FrontModule\Model\Subscribers;

use Core\Model\Event\ControlEvent;
use Ecomail\Model\EcomailConfig;
use Ecomail\Model\EcomailOrders;
use EshopOrders\FrontModule\Components\Order\OrderForm;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\OrderFlag;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Core\Model\Entities\EntityManagerDecorator;
use Tracy\Debugger;

class OrderFormSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected EcomailOrders $ecomailOrders;

	public function __construct(EntityManagerDecorator $em, EcomailOrders $ecomailOrders)
	{
		$this->em            = $em;
		$this->ecomailOrders = $ecomailOrders;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			OrderForm::class . '::onAttach' => 'orderFormAttach',
			'eshopOrders.orderOnSuccess'    => 'orderOnSuccess',
		];
	}

	public function orderFormAttach(ControlEvent $event): void
	{
		/** @var OrderForm $control */
		$control = $event->control;

		$control->allowNewsletterFlag();
	}

	public function orderOnSuccess(OrderEvent $event): void
	{
		if (!EcomailConfig::load('apiKey')) {
			return;
		}

		try {
			$order = $event->order;
			/** @var OrderFlag|null $newsletterFlag */
			$newsletterFlag = $order->getOrderFlags()->get(OrderFlag::TYPE_NEWSLETTER);

			if (!$newsletterFlag || $newsletterFlag->getState() != 1) {
				return;
			}

			$this->ecomailOrders->exportOrder($order);
		} catch (\Exception $e) {
			Debugger::log('Ecomail order error: ' . $e->getMessage(), 'ecomail');
		}
	}
}
