<?php declare(strict_types = 1);

namespace Ecomail\Model;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Ecomail\Model\Dao\ListSubscribe;
use Ecomail\Model\Dao\Transaction;
use Ecomail\Model\Dao\TransactionItem;
use Ecomail\Model\Entities\EcomailOrder;
use Ecomail\Model\Helpers\OrderHelper;
use EshopCatalog\Model\Entities\CategoryTexts;
use EshopCatalog\Model\Entities\ProductInSite;
use EshopOrders\Model\Entities\Order;
use Tracy\Debugger;

class EcomailOrders
{
	protected EntityManagerDecorator $em;
	protected Ecomail                $ecomail;
	protected Translator             $translator;

	public function __construct(
		EntityManagerDecorator $em,
		Ecomail                $ecomail,
		Translator             $translator
	)
	{
		$this->em         = $em;
		$this->ecomail    = $ecomail;
		$this->translator = $translator;
	}

	public function exportOrder(Order $order): void
	{
		$repo   = $this->em->getRepository(EcomailOrder::class);
		$entity = $repo->find($order->getId());

		if (!$entity) {
			$entity = new EcomailOrder($order);
		}

		if (!$entity->getExported()) {
			$confLists = EcomailConfig::load('lists');
			$confList  = $confLists[$order->site->getIdent()] ?? $confLists['default'];

			$subscriber = OrderHelper::fillSubscriber($order->getAddressInvoice());

			$list                        = new ListSubscribe($subscriber, (int) $confList['id']);
			$list->triggerAutoresponders = $confList['triggerAutoresponders'];
			$list->updateExisting        = $confList['updateExisting'];
			$list->resubscribe           = $confList['resubscribe'];
			$list->skipConfirmation      = $confList['skipConfirmation'];
			$list->triggerNotification   = $confList['triggerNotification'];

			$customer = $order->getCustomer();
			if ($customer && $confList['addCustomerToGroup']) {
				$customerGroup = $customer->getGroupCustomers() ? $customer->getGroupCustomers()->getId() : null;

				foreach ($confList['addCustomerToGroup'] as $groupKey => $data) {
					if ($customerGroup) {
						if (isset($data[$customerGroup])) {
							$list->groups[$groupKey] = $data[$customerGroup];
						}
					} else {
						if (isset($data['no'])) {
							$list->groups[$groupKey] = $data['no'];
						}
					}
				}
			}

			$r = (array) $this->ecomail->addSubscriber($list);

			// Pokud se odešle uživatel, odešleme i data objednávky
			if (isset($r['id'])) {
				$transaction          = new Transaction(
					(string) $order->getId(),
					$subscriber->email,
					$order->site->getIdent(),
					$order->getSpedition()->getPrice() + $order->getPayment()->getPrice(),
				);
				$transaction->city    = $subscriber->city;
				$transaction->country = $subscriber->country;

				foreach ($order->getOrderItems() as $item) {
					/** @var ProductInSite|null $productInSite */
					$productInSite = $item->getProduct() && $item->getProduct()->sites->get($order->site->getIdent())
						? $item->getProduct()->sites->get($order->site->getIdent())
						: null;
					$category      = $productInSite->category;

					if ($category) {
						/** @var CategoryTexts|null $categoryText */
						$categoryText = $category->getCategoryTexts()->toArray()[$this->translator->getLocale()] ?? $category->getCategoryTexts()->first();
					} else {
						$categoryText = null;
					}

					$transactionItem = new TransactionItem(
						(string) $item->getId(),
						$item->getOrderItemText()->getName(),
						$categoryText ? (string) $categoryText->getName() : '',
						$item->getPrice(),
						$item->getQuantity(),
					);

					$transaction->addItem($transactionItem);
				}

				$r2 = (array) $this->ecomail->createNewTransaction($transaction);

				if (isset($r2['id'])) {
					$entity->export();
					$this->em->persist($entity);
					$this->em->flush();
				} else {
					if (isset($r2['error'])) {
						Debugger::log('Ecomail create new transaction error: ' . $r2['error'], 'ecomail');
					}
				}
			} else {
				if (isset($r['error'])) {
					Debugger::log('Ecomail add Subscriber error: ' . $r['error'], 'ecomail');
				}
			}
		}
	}
}
