<?php declare(strict_types=1);

namespace Eet\Model\Providers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Eet\Model\EetConfig;
use Eet\Model\Entities\OrderReceipt;
use Eet\Model\Entities\SellerReceiptConfig;
use Eet\Model\Exceptions\EetTransactionFailed;
use Eet\Model\Exceptions\SellerDataNotFilled;
use Eet\Model\Exceptions\SellerDataNotFilledException;
use Eet\Model\Repository\SellerReceiptConfigs;
use EshopCatalog\AdminModule\Model\Sellers;
use EshopCatalog\Model\Entities\Seller;
use EshopOrders\Model\Entities\Order;
use Nette\Localization\ITranslator;
use Nette\Schema\Expect;
use Nette\Schema\Processor;
use Nette\Schema\ValidationException;
use Nette\Utils\Arrays;
use Nette\Utils\DateTime;
use SlevomatEET\Client;
use SlevomatEET\Configuration;
use SlevomatEET\Cryptography\CryptographyService;
use SlevomatEET\Driver\GuzzleSoapClientDriver;
use SlevomatEET\EvidenceEnvironment;
use SlevomatEET\Receipt;
use SlevomatEET\FailedRequestException;
use SlevomatEET\InvalidResponseReceivedException;
use Tracy\Debugger;

class EetProvider
{
	/** @var Order */
	protected Order $order;

	/** @var EntityManagerDecorator */
	protected EntityManagerDecorator $em;

	/** @var SellerReceiptConfigs */
	protected SellerReceiptConfigs $sellerReceiptConfigs;

	/** @var Sellers */
	protected Sellers $sellers;

	/** @var ITranslator */
	protected ITranslator $translator;

	/** @var array */
	protected array $data = [
		'vatId' => null,
		'premiseId' => null,
		'cashRegisterId' => null,
		'delegatedVatId' => null,
		'receiptNumber' => null,
		'totalPrice' => null,
	];

	/**
	 * EetProvider constructor.
	 * @param Order $order
	 * @param EntityManagerDecorator $em
	 * @param SellerReceiptConfigs $sellerReceiptConfigs
	 * @param Sellers $sellers
	 * @param ITranslator $translator
	 */
	public function __construct(Order $order, EntityManagerDecorator $em, SellerReceiptConfigs $sellerReceiptConfigs,
	                            Sellers $sellers, ITranslator $translator)
	{
		$this->order = $order;
		$this->em = $em;
		$this->sellerReceiptConfigs = $sellerReceiptConfigs;
		$this->sellers = $sellers;
		$this->translator = $translator;
		$this->setData($order);
	}

	/**
	 * @param bool $firstSend
	 * @throws \Exception
	 */
	public function send(bool $firstSend = true): void
	{
		$this->validateData();

		$crypto = $this->getCryptographyService();

		$configuration = new Configuration(
			$this->data['vatId'],
			$this->data['premiseId'],
			$this->data['cashRegisterId'],
			EvidenceEnvironment::get($this->getEnvironment()),
			EetConfig::load('verificationMode', false)
		);
		$client = new Client($crypto, $configuration, new GuzzleSoapClientDriver(new \GuzzleHttp\Client()));

		$receiptTime = new \DateTimeImmutable();
		$receipt = new Receipt(
			$firstSend, // TODO umoznit opakovat zaslani - OVERIT ZE SE NEZVEDA COUNTER uctenky pri opakovani
			$this->data['delegatedVatId'],
			(string) $this->data['receiptNumber'],
			$receiptTime,
			$this->data['totalPrice']
		);

		$fik = $bkp = $pkp = null;
		$isOk = true;

		try {
			$response = $client->send($receipt);
			$fik = $response->getFik();
			$bkp = $response->getBkp();
		} catch (FailedRequestException $e) {
			$pkp = $e->getRequest()->getPkpCode();
			$bkp = $e->getRequest()->getBkpCode();
			$isOk = false;
			$this->log($e);
		} catch (InvalidResponseReceivedException $e) {
			$pkp = $e->getResponse()->getRequest()->getPkpCode();
			$bkp = $e->getResponse()->getRequest()->getBkpCode();
			$isOk = false;
			$this->log($e);
		}

		$orderReceipt = new OrderReceipt(
			$this->order, $isOk,
			$this->data['receiptNumber'], $this->data['cashRegisterId'], $this->data['premiseId'],
			$bkp, $pkp, $fik
		);

		$this->em->persist($orderReceipt);
		$this->em->flush();

		if (!$isOk) {
			throw new EetTransactionFailed('eet.messages.submissionFailed');
		}
	}

	/**
	 * @param \Exception $exception
	 */
	protected function log($exception): void
	{
		$now = new DateTime;
		Debugger::log($exception, 'eet-%s-%s', $now->format('Y'), $now->format('m'));
		LogNotifier::toDevelopers($exception->getMessage(), 'Chyba v EET');
	}

	/**
	 * @param Order $order
	 */
	protected function setData(Order $order): void
	{
		/** @var Seller $seller */
		$seller = $this->sellers->getSellerForSite($this->order->site->getIdent());

		if (!$seller) {
			return;
		}

		/** @var SellerReceiptConfig $receiptConfig */
		$receiptConfig = $this->sellerReceiptConfigs->getBySeller($seller->getId());

		if (!$receiptConfig) {
			return;
		}

		$this->data['vatId'] = $seller->getDic();
		$this->data['premiseId'] = $receiptConfig->premise;
		$this->data['cashRegisterId'] = $receiptConfig->checkout;
		$this->data['delegatedVatId'] = $seller->getDic();
		$this->data['receiptNumber'] = $receiptConfig->counter + 1;
		$this->data['totalPrice'] = (int) ($order->getPrice() * 100); // eg. 55.5,- -> 5550,-
	}

	/**
	 * @return string
	 */
	protected function getEnvironment(): string
	{
		return CORE_TEST_MODE ? EvidenceEnvironment::PLAYGROUND : EvidenceEnvironment::PRODUCTION;
	}

	/**
	 * @return CryptographyService
	 */
	protected function getCryptographyService(): CryptographyService
	{
		if ($this->getEnvironment() === EvidenceEnvironment::PLAYGROUND) {
			$keyPath = sprintf('%s/%s/%s/%s/%s', SRC_DIR, '..', 'slevomat', 'eet-client', 'cert');
			$privateKey = sprintf('%s/%s', $keyPath, 'EET_CA1_Playground-CZ00000019.key');
			$publicKey = sprintf('%s/%s', $keyPath, 'EET_CA1_Playground-CZ00000019.pub');
			$password = '';
		} else {
			// TODO
		}

		return new CryptographyService($privateKey, $publicKey, $password);
	}

	/**
	 * @return bool
	 * @throws EetTransactionFailed
	 */
	protected function validateData(): void
	{
		$processor = new Processor;
		$schema = Expect::structure([
			'vatId' => Expect::string()->required(),
			'premiseId' => Expect::string()->required(),
			'cashRegisterId' => Expect::string()->required(),
			'delegatedVatId' => Expect::string()->required(),
			'receiptNumber' => Expect::int()->required(),
			'totalPrice' => Expect::int()->required(),
		]);

		try {
			$processor->process($schema, $this->data);
		} catch (ValidationException $exception) {
			throw new SellerDataNotFilledException($this->translator->translate('eet.messages.notFilledSellerData'));
		}
	}

}