<?php declare(strict_types=1);

namespace Eet\Model\Subscribers;

use Contributte\EventDispatcher\EventDispatcher;
use Eet\Model\Events\OrderPaidEvent;
use Core\AdminModule\Components\Navigation\Navigation;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Event\ControlEvent;
use Eet\AdminModule\Components\EetStatus;
use Eet\AdminModule\Components\IEetStatusFactory;
use Eet\Model\EetConfig;
use Eet\Model\Repository\OrderReceipts;
use EshopOrders\AdminModule\Components\Order\OrderForm;
use EshopOrders\AdminModule\Model\Event\OrderNavigationCallbackEvent;
use EshopOrders\AdminModule\Model\Event\OrdersNavigationEvent;
use EshopOrders\Model\Orders;
use Nette\Application\LinkGenerator;
use Nette\Application\Request;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderFormSubscriber implements EventSubscriberInterface
{
	protected IEetStatusFactory $eetStatusFactory;
	protected LinkGenerator $linkGenerator;
	protected OrderReceipts $orderReceipts;
	protected EventDispatcher $eventDispatcher;
	protected Orders $orders;
	protected const SIGNAL_KEY = 'generateEet';

	/**
	 * OrderFormSubscriber constructor.
	 * @param IEetStatusFactory $eetStatusFactory
	 * @param OrderReceipts $orderReceipts
	 * @param LinkGenerator $linkGenerator
	 * @param Orders $orders
	 */
	public function __construct(IEetStatusFactory $eetStatusFactory, OrderReceipts $orderReceipts, LinkGenerator $linkGenerator,
	                            Orders $orders, EventDispatcher $eventDispatcher)
	{
		$this->eetStatusFactory = $eetStatusFactory;
		$this->linkGenerator = $linkGenerator;
		$this->orderReceipts = $orderReceipts;
		$this->orders = $orders;
		$this->eventDispatcher = $eventDispatcher;
	}

	/**
	 * @return array
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.admin.onBeforeRenderEditOrderNavigation' => ['onRenderNavigation', 100],
			'eshopOrders.admin.ordersDefaultNavigationCallback' => ['onRenderNavigationCallback', 100],
			sprintf('%s::onAttach', OrderForm::class) => ['onRenderOrderForm', 100]
		];
	}

	/**
	 * @param ControlEvent $event
	 */
	public function onRenderOrderForm(ControlEvent $event): void
	{
		if (!EetConfig::load('enable', false)) {
			return;
		}

		$orderId = (int) $event->control->getPresenter()->getParameter('id');
		$event->control->addWidgetControl('eetStatus', $this->eetStatusFactory->create($orderId));
	}

	/**
	 * @param ControlEvent $event
	 */
	public function onRenderNavigation(OrdersNavigationEvent $event): void
	{
		if (!EetConfig::load('enable', false)) {
			return;
		}

		$event->addNavigation(
			'eet.navigation.sendEet',
			['navigationCallback!', ['key' => self::SIGNAL_KEY]],
			'fas fa-paper-plane',
			'ajax bg-info',
			['Eet:Admin' => 'access']
		);
	}

	/**
	 * @param OrderNavigationCallbackEvent $event
	 */
	public function onRenderNavigationCallback(OrderNavigationCallbackEvent $event): void
	{
		// TODO do eshopCheckout
		$orderId = (int) $event->presenter->getParameter('id');

		if ($event->key !== self::SIGNAL_KEY) {
			return;
		}

		$hasOrderReceipt = $this->orderReceipts->hasOrderReceipt($orderId);

		// order receipt does not exists
		if (!$hasOrderReceipt) {
			$event->presenter->flashMessageWarning('eet.editOrder.messages.notCreated');
			$event->presenter->redrawControl('flashes');
			return;
		}

		// order does not have status paid in cash
		if (!$this->orderReceipts->meetsRequirementsForOrderReceipt($orderId)) {
			$event->presenter->flashMessageWarning('eet.editOrder.messages.warning');
			$event->presenter->redrawControl('flashes');
			return;
		}

		if (!$order = $this->orders->getReference($orderId)) {
			return;
		}

		try {
			// TODO povolit znovu odeslat uctenku, pokud byla ve spatnem stavu
			// order has receipt -> not recreate
			//		if (!$hasOrderReceipt || $orderReceiptConfig->is) {
			if ($hasOrderReceipt) {
				$this->eventDispatcher->dispatch(new OrderPaidEvent($order), 'eet.orderPaidInCash');
			}

			$event->presenter->redirect(':Eet:Admin:OrderReceipts:Pdf', $orderId);
		} catch (\Exception $exception) {
			$event->presenter->flashMessageDanger($exception->getMessage());
			$event->presenter->redrawControl('flashes');
		}

	}

}