<?php declare(strict_types=1);

namespace Eet\Model\Subscribers;

use Core\AdminModule\Components\Navigation\Navigation;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\RenderNavigationEvent;
use Eet\AdminModule\Components\EetStatus;
use Eet\AdminModule\Components\IEetStatusFactory;
use Eet\Model\EetConfig;
use Eet\Model\Entities\SellerReceiptConfig;
use Eet\Model\Repository\SellerReceiptConfigs;
use EshopCatalog\AdminModule\Model\Event\SellerEvent;
use EshopCatalog\AdminModule\Model\Event\SellerFormEvent;
use EshopCatalog\AdminModule\Model\Event\SetSellerFormDataEvent;
use EshopOrders\AdminModule\Components\Order\OrderForm;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class SellerFormSubscriber implements EventSubscriberInterface
{
	/** @var string */
	protected const FORM_CONTAINER_NAME = 'eet';

	/** @var SellerReceiptConfigs */
	protected SellerReceiptConfigs $receiptConfigs;

	/** @var EntityManagerDecorator */
	protected EntityManagerDecorator $em;

	/**
	 * SellerFormSubscriber constructor.
	 * @param SellerReceiptConfigs $receiptConfigs
	 * @param EntityManagerDecorator $em
	 * @param IEetStatusFactory $eetStatusFactory
	 */
	public function __construct(SellerReceiptConfigs $receiptConfigs, EntityManagerDecorator $em, IEetStatusFactory $eetStatusFactory)
	{
		$this->receiptConfigs = $receiptConfigs;
		$this->em = $em;
		$this->eetStatusControl = $eetStatusFactory;
	}

	/**
	 * @return array
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'eshopCatalog.createSellerForm' => ['createSellerForm', 100],
			'eshopCatalog.setSellerFormData' => ['setSellerFormData', 100],
			'eshopCatalog.sellerBeforeSave' => ['sellerBeforeSave', 100]
		];
	}

	/**
	 * @param SellerFormEvent $event
	 */
	public function createSellerForm(SellerFormEvent $event): void
	{
		if (!$this->isEetAllowed()) {
			return;
		}

		$container = $event->form['extendedContainers']->addContainer(self::FORM_CONTAINER_NAME, 'eet.sellerForm.title');
		$container->addText('premise', 'eet.sellerForm.premise')
				  ->setRequired();
		$container->addText('checkout', 'eet.sellerForm.checkout')
				  ->setRequired();
		$event->form['dic']->setRequired();
	}

	/**
	 * @param SetSellerFormDataEvent $event
	 */
	public function setSellerFormData(SetSellerFormDataEvent $event): void
	{
		if (!$this->isEetAllowed()) {
			return;
		}

		$config = $this->receiptConfigs->get($event->seller->getId());

		if (!$config) {
			return;
		}

		$defaults['extendedContainers'][self::FORM_CONTAINER_NAME]['premise'] = $config->premise;
		$defaults['extendedContainers'][self::FORM_CONTAINER_NAME]['checkout'] = $config->checkout;
		$event->form->setDefaults($defaults);
	}

	/**
	 * @param SellerEvent $event
	 */
	public function sellerBeforeSave(SellerEvent $event): void
	{
		if (!$this->isEetAllowed()) {
			return;
		}

		$containerData = $event->formData->extendedContainers->{self::FORM_CONTAINER_NAME};
		$premise = $containerData->premise;
		$checkout = $containerData->checkout;

		if ($event->seller->getId()) {
			$config = $this->receiptConfigs->get($event->seller->getId());

			if ($config) {
				$config->checkout = $checkout;
				$config->premise = $premise;

				$this->em->persist($config);
				return;
			}
		}

		$this->em->persist(new SellerReceiptConfig($event->seller, $checkout, $premise));
	}

	/**
	 * @return bool
	 */
	protected function isEetAllowed(): bool
	{
		return EetConfig::load('enable', false);
	}

}