<?php declare(strict_types = 1);

namespace EshopAccountant\AdminModule\Components;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Helpers\Strings;
use Core\Model\Http\CsvResponse;
use Core\Model\Templating\Filters\Price;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopAccountant\Model\EshopAccountantConfig;
use EshopCatalog\AdminModule\Model\Sellers;
use EshopCatalog\AdminModule\Model\VatRates;
use EshopCatalog\Model\Config;
use EshopOrders\AdminModule\Model\DataConverter;
use EshopOrders\AdminModule\Model\GroupsCustomers;
use EshopOrders\AdminModule\Model\Payments as PaymentsAdmin;
use EshopOrders\FrontModule\Model\Payments;
use EshopOrders\FrontModule\Model\Speditions;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Orders;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;

class ExportForm extends BaseControl
{
	protected Invoices        $invoices;
	protected Speditions      $speditions;
	protected Payments        $payments;
	protected PaymentsAdmin   $paymentsAdmin;
	protected DataConverter   $dataConverter;
	protected Sellers         $sellers;
	protected Orders          $orders;
	protected VatRates        $vatRates;
	protected GroupsCustomers $groupsCustomers;

	public function __construct(
		Invoices        $invoices,
		Speditions      $speditions,
		Payments        $payments,
		PaymentsAdmin   $paymentsAdmin,
		DataConverter   $dataConverter,
		Sellers         $sellers,
		Orders          $orders,
		VatRates        $vatRates,
		GroupsCustomers $groupsCustomers
	)
	{
		$this->invoices        = $invoices;
		$this->speditions      = $speditions;
		$this->payments        = $payments;
		$this->paymentsAdmin   = $paymentsAdmin;
		$this->dataConverter   = $dataConverter;
		$this->sellers         = $sellers;
		$this->orders          = $orders;
		$this->vatRates        = $vatRates;
		$this->groupsCustomers = $groupsCustomers;
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false);

		$now = new DateTime();

		$form->addDatePicker('from', 'eshopAccountant.export.from')
			->setDefaultValue((clone $now)->modify('first day of this month'));
		$form->addDatePicker('to', 'eshopAccountant.export.to')
			->setDefaultValue((clone $now)->modify('last day of this month'));
		$form->addSelect('invoiceByDate', 'eshopAccountant.export.invoiceByDate', [
			'default'     => 'eshopAccountant.export.dates.invoiceCreated',
			'invoiceDUZP' => 'eshopAccountant.export.dates.invoiceDUZP',
		])->setDefaultValue('orderCreated');
		$form->addSelect('seller', 'eshopAccountant.export.seller',
			['' => $this->t('eshopAccountant.export.allSellers')] + $this->sellers->getOptionsForSelect())
			->setTranslator(null);
		$form->addSelect('isPaid', $this->t('eshopAccountant.export.isPaid'), [
			''  => $this->t('eshopAccountant.export.all'),
			'1' => $this->t('eshopAccountant.export.paid'),
			'0' => $this->t('eshopAccountant.export.unpaid'),
		])->setTranslator(null);
		$form->addSelect('payment', $this->t('eshopAccountant.export.payment'), [
				'' => $this->t('eshopAccountant.export.all'),
			] + $this->paymentsAdmin->getForSelectOption())
			->setTranslator(null);
		$form->addSelect('groupCustomer', $this->t('eshopAccountant.export.groupCustomer'), [
				'' => $this->t('eshopAccountant.export.all'),
				0  => $this->t('eshopAccountant.export.withoutGroup'),
			] + $this->groupsCustomers->getOptionsForSelect())
			->setTranslator(null);

		$form->addCheckbox('withoutODD', 'eshopAccountant.export.withoutODD');

		if (EshopAccountantConfig::load('allowExportInvoices')) {
			$form->addSubmit('exportXml', 'eshopAccountant.export.exportXml');
			$form->addSubmit('invoicesCheckList', 'eshopAccountant.export.invoicesCheckList');
		}

		if (EshopAccountantConfig::load('allowExportOrders')) {
			$form->addSubmit('exportOrdersXml', 'eshopAccountant.export.exportOrdersXml');
			$form->addSubmit('exportPdf', 'eshopAccountant.export.exportPdf');
			if (EshopAccountantConfig::load('allowExportOrdersToCsv')) {
				$form->addSubmit('exportCsv', 'eshopAccountant.export.exportCsv');
			}
		}

		$form->onSuccess[] = [$this, 'formSuccess'];

		$this->eventDispatcher->dispatch(
			new CreateFormEvent($form, $this->getPresenterIfExists() ? $this->template : null),
			static::class . '::createForm'
		);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		$values->from          = DateTime::from($values->from)->setTime(0, 0, 0);
		$values->to            = DateTime::from($values->to)->setTime(23, 59, 59);
		$values->seller        = $values->seller ? (int) $values->seller : null;
		$values->isPaid        = $values->isPaid === '' ? null : (int) $values->isPaid;
		$values->payment       = $values->payment === '' ? null : (int) $values->payment;
		$values->groupCustomer = $values->groupCustomer === '' ? null : (int) $values->groupCustomer;

		$vatRatesPohoda = Config::load('vatRate.showPohodaField', false) ? $this->vatRates->getListForPohoda() : [];

		if ($form->getComponent('exportXml')->isSubmittedBy() && EshopAccountantConfig::load('allowExportInvoices')) {
			$invoices = $this->invoices->getByDateAndSeller(
				$values->seller,
				$values->from,
				$values->to,
				$values->isPaid,
				$values->payment,
				$values->groupCustomer,
				$values->invoiceByDate,
				(int) $values->withoutODD
			);
			$sellers  = $this->sellers->getAll();

			$response = $this->getPresenter()->getHttpResponse();
			$response->setContentType('text/xml');
			$response->addHeader('Content-Disposition', 'attachment; filename="export.xml"');

			$template = clone $this->template;
			echo $template->renderToString(__DIR__ . '/ExportForm_xml.latte', [
				'sellers'    => $sellers,
				'invoices'   => $invoices,
				'vatRates'   => $vatRatesPohoda,
				'speditions' => $this->speditions->getAllPublished(),
				'payments'   => $this->payments->getAllPublished(),
			]);

			exit;
		} else if ($form->getComponent('exportOrdersXml')->isSubmittedBy() && EshopAccountantConfig::load('allowExportOrders')) {
			$orders  = $this->orders->getByDateAndSeller(
				$values->seller,
				$values->from,
				$values->to,
				$values->isPaid,
				$values->payment,
				$values->groupCustomer,
				(int) $values->withoutODD
			);
			$sellers = $this->sellers->getAll();

			$response = $this->getPresenter()->getHttpResponse();
			$response->setContentType('text/xml');
			$response->addHeader('Content-Disposition', 'attachment; filename="export.xml"');

			$template = clone $this->template;
			echo $template->renderToString(__DIR__ . '/ExportForm_ordersXml.latte', [
				'sellers'  => $sellers,
				'orders'   => $orders,
				'vatRates' => $vatRatesPohoda,
			]);

			exit;
		} else if ($form->getComponent('exportPdf')->isSubmittedBy() && EshopAccountantConfig::load('allowExportOrders')) {
			$result = $this->dataConverter->forPrint($this->invoices->getOrdersByDateAndSeller(
				$values->seller,
				$values->from,
				$values->to,
				$values->isPaid,
				$values->payment,
				$values->groupCustomer,
				(int) $values->withoutODD
			));

			$data        = [];
			$totalGroups = [];
			foreach ($result['orders'] as $order) {
				$data['byPayment'][$order['paymentName']][] = $order;

				if (!isset($totalGroups['byPayment'][$order['paymentName']])) {
					foreach ($result['vatTotal']['vats'] as $k => $v) {
						$totalGroups['byPayment'][$order['paymentName']]['total']    = 0;
						$totalGroups['byPayment'][$order['paymentName']]['vats'][$k] = [
							'base'  => 0,
							'dph'   => 0,
							'total' => 0,
						];
					}
				}

				foreach ($order['vats'] as $k => $v) {
					$totalGroups['byPayment'][$order['paymentName']]['total']             += $v['total'];
					$totalGroups['byPayment'][$order['paymentName']]['vats'][$k]['base']  += $v['base'];
					$totalGroups['byPayment'][$order['paymentName']]['vats'][$k]['dph']   += $v['dph'];
					$totalGroups['byPayment'][$order['paymentName']]['vats'][$k]['total'] += $v['total'];
				}
			}

			krsort($result['vatTotal']['vats']);
			foreach ($totalGroups['byPayment'] as $k => $v) {
				krsort($totalGroups['byPayment'][$k]['vats']);
			}

			Price::$forceDecimals   = 2;
			Strings::$priceKeepTail = true;
			$template               = clone $this->template;
			echo $template->renderToString(__DIR__ . '/ExportForm_pdf.latte', [
				'data'        => $data,
				'vatTotal'    => $result['vatTotal'],
				'statuses'    => $result['statuses'],
				'totalGroups' => $totalGroups,
			]);
			Price::$forceDecimals   = null;
			Strings::$priceKeepTail = false;

			exit;
		} else if ($form->getComponent('exportCsv')->isSubmittedBy() && EshopAccountantConfig::load('allowExportOrdersToCsv')) {
			$orders  = $this->orders->getByDateAndSeller(
				$values->seller,
				$values->from,
				$values->to,
				$values->isPaid,
				$values->payment,
				$values->groupCustomer,
				(int) $values->withoutODD
			);
			$sellers = $this->sellers->getAll();

			$template    = clone $this->template;
			$strTemplate = $template->renderToString(__DIR__ . '/ExportForm_csv.latte', [
				'sellers' => $values->seller ? array_filter($sellers, fn($seller) => $seller->getId() === $values->seller) : $sellers,
				'orders'  => $orders,
			]);

			CsvResponse::sendResponse('export.csv', $strTemplate);
		} else if ($form->getComponent('invoicesCheckList')->isSubmittedBy() && EshopAccountantConfig::load('allowExportInvoices')) {
			$result = $this->invoices->getCheckList(
				$values->seller,
				$values->from,
				$values->to,
				$values->isPaid,
				$values->payment,
				$values->groupCustomer,
				$values->invoiceByDate,
				(int) $values->withoutODD
			);

			Price::$forceDecimals   = 2;
			Strings::$priceKeepTail = true;
			$template               = clone $this->template;
			echo $template->renderToString(__DIR__ . '/ExportForm_checkList.latte', [
				'list'    => $result['list'],
				'total'   => $result['total'],
				'items'   => $result['items'],
				'allVats' => array_unique($result['allVats']),
			]);
			Price::$forceDecimals   = null;
			Strings::$priceKeepTail = false;

			exit;
		}

		$event = new FormSuccessEvent($form, $values, $this->template, $this->presenter);
		$this->eventDispatcher->dispatch($event, static::class . '::formSuccess');
	}
}
