<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\AdminModule\Components\Group;

use EshopAdvancedFeature\AdminModule\Model\VirtualCategoryGroups;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryGroup;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryGroupFilter;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryGroupText;
use EshopCatalog\Model\Entities\Feature;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class GroupForm extends BaseControl
{
	/** @var int|null @persistent */
	public ?int $id = null;

	public ?VirtualCategoryGroup $group = null;

	protected VirtualCategoryGroups $virtualCategoryGroups;

	public function __construct(
		VirtualCategoryGroups $virtualCategoryGroups
	)
	{
		$this->virtualCategoryGroups = $virtualCategoryGroups;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$filters = [];
		foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')->select('f.id, ft.name')
			         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
			         ->andWhere('f.useAsFilter = 1')
			         ->setParameters(['lang' => $this->translator->getLocale()])->orderBy('f.position')
			         ->getQuery()->getArrayResult() as $row) {
			$filters[$row['id']] = $row['name'];
		}

		$titlePrefix = 'eshopAdvancedFeature.virtualCategoryGroup.';
		$form->addText('title', $titlePrefix . 'title')
			->setIsMultilanguage();
		$form->addFilesManager('icon', $titlePrefix . 'icon');
		$form->addSortableCheckboxList('filters', $this->t('eshopCatalog.categoryForm.availableFilters'), $filters)->setTranslator(null);

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var VirtualCategoryGroupText[] $texts */
			$texts = [];

			if ($this->id) {
				$group = $this->virtualCategoryGroups->get($this->id);

				if (!$group) {
					throw new \InvalidArgumentException();
				}

				$texts = $group->texts->toArray();
			} else {
				$group = new VirtualCategoryGroup();
			}

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l]) && $v['title']) {
					$texts[$l] = new VirtualCategoryGroupText($group, $l, $v['title']);
				}

				if (!$v['title'] || $v['title'] == '') {
					if (isset($texts[$l])) {
						$this->em->remove($texts[$l]);
					}

					unset($texts[$l]);
					continue;
				}

				$texts[$l]->title = $v['title'];

				$this->em->persist($texts[$l]);
				$group->texts->set($l, $texts[$l]);
			}

			$group->icon = $values->icon;

			// FILTRY
			$formFilters  = $values->filters;
			$groupFilters = $group->filters->toArray();

			// nové
			foreach (array_diff($formFilters, array_keys($groupFilters)) as $k => $v) {
				$groupFilter           = new VirtualCategoryGroupFilter($group, $this->em->getReference(Feature::class, $v));
				$groupFilter->position = $k;

				$this->em->persist($groupFilter);
				$group->filters->set($v, $groupFilter);
			}

			// odstranit
			foreach (array_diff(array_keys($groupFilters), $formFilters) as $v) {
				$groupFilter = $group->filters->get($v);
				if ($groupFilter) {
					$group->filters->remove($v);
					$this->em->remove($groupFilter);
				}
			}

			// seřadit
			foreach ($formFilters as $k => $v) {
				$groupFilter = $group->filters->get($v);
				if ($groupFilter) {
					$groupFilter->position = $k;
					$this->em->persist($groupFilter);
				}
			}

			$this->em->persist($group);
			$this->em->flush();

			$form->addCustomData('featureNavigationGroup', $group->getId());
			$this->getPresenter()->flashMessageSuccess('default.saved');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setVirtualCategoryGroup(int $id): void
	{
		$this->id = $id;
		$group    = $this->virtualCategoryGroups->get($id);

		if (!$group) {
			throw new InvalidArgumentException('');
		}

		$this->group = $group;

		$d = [
			'icon'    => $group->icon,
			'filters' => $group->filters->getKeys(),
		];

		foreach ($group->texts->toArray() as $lang => $vals) {
			foreach (['title'] as $v) {
				$d[$v][$lang] = $vals->{$v};
			}
		}

		$this['form']->setDefaults($d);
	}
}
