<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\AdminModule\Components\VirtualCategory;

use Core\Model\Entities\QueryBuilder;
use Core\Model\Helpers\Arrays;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategories;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategoryGroups;
use EshopAdvancedFeature\Model\EshopAdvancedFeatureConfig;
use EshopAdvancedFeature\Model\Entities\VirtualCategory;
use Nette\Application\UI\Multiplier;
use Nette\Utils\Html;

class VirtualCategoriesGrid extends BaseControl
{
	protected string                      $siteIdent;
	protected VirtualCategories           $virtualCategories;
	protected VirtualCategoryGroups       $virtualCategoryGroups;
	protected IVirtualCategoryFormFactory $formFactory;
	protected IRelatedFormFactory         $relatedFormFactory;

	public function __construct(
		string                      $siteIdent,
		VirtualCategories           $virtualCategories,
		VirtualCategoryGroups       $virtualCategoryGroups,
		IVirtualCategoryFormFactory $formFactory,
		IRelatedFormFactory         $relatedFormFactory
	)
	{
		$this->siteIdent             = $siteIdent;
		$this->virtualCategories     = $virtualCategories;
		$this->virtualCategoryGroups = $virtualCategoryGroups;
		$this->formFactory           = $formFactory;
		$this->relatedFormFactory    = $relatedFormFactory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ============= Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle       = $this->t('eshopAdvancedFeature.title.addVirtualCategory');
		$this->template->modal            = 'form';
		$this->template->modalDialogClass = 'modal-xxl';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this['form']->id                 = $id;
		$this->template->modalTitle       = $this->t('eshopAdvancedFeature.title.editVirtualCategory',
			['title' => $this['form']->getVirtualCategory()->getText()->h1]);
		$this->template->modal            = 'form';
		$this->template->modalDialogClass = 'modal-xxl';
		$this->redrawControl('modal');
	}

	public function handleDelete($id): void
	{
		$presenter = $this->getPresenter();

		if (is_array($id)
			? $this->virtualCategories->removeMultiple($id)
			: $this->virtualCategories->remove($id)) {
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		$presenter->redrawControl('flashes');
		$this['grid']->reload();
	}

	public function handleRelated(int $id): void
	{
		$virtualCategory = $this->virtualCategories->get($id);

		if ($virtualCategory) {
			$this->template->modalTitle       = $this->t('eshopAdvancedFeature.title.editRelated', ['title' => trim($virtualCategory->getText()->pageTitle . ' (' . $virtualCategory->getText()->getUrl() . ')')]);
			$this->template->modal            = 'relatedForm';
			$this->template->modalDialogClass = 'modal-xl';
			$this->template->id               = $id;
			$this->redrawControl('modal');
		}
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ============= Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->virtualCategories->getEr()->createQueryBuilder('vc')
			->addSelect('t, inG, g, gText')
			->innerJoin('vc.texts', 't', Join::WITH, 't.locale = :locale')
			->leftJoin('vc.inGroups', 'inG')
			->leftJoin('inG.group', 'g')
			->leftJoin('g.texts', 'gText', Join::WITH, 'gText.lang = :locale')
			->andWhere('vc.siteIdent = :siteIdent')
			->setParameters([
				'locale'    => $this->translator->getLocale(),
				'siteIdent' => $this->siteIdent,
			])
			->addOrderBy('vc.created', 'DESC');

		$grid->setDataSource($qb);

		$columnPrefix = 'eshopAdvancedFeature.virtualCategory.';

		// Toolbar
		$grid->addToolbarButton('add!', 'default.add')
			->setIcon('plus')
			->setClass('ajax btn-success btn');

		// Columns
		$grid->addColumnText('url', $columnPrefix . 'url', 'text.url')
			->setFilterText()
			->setCondition(static function(QueryBuilder $qb, string $value) {
				if (!Arrays::contains($qb->getAllAliases(), 'fText')) {
					$qb->innerJoin('vc.texts', 'fText');
				}

				$qb->andWhere('fText.url LIKE :url')
					->setParameter('url', "%$value%");
			});
		$grid->addColumnText('h1', $columnPrefix . 'h1', 'text.h1')
			->setFilterText()
			->setCondition(function(QueryBuilder $qb, string $value) {
				if (!Arrays::contains($qb->getAllAliases(), 'fText')) {
					$qb->innerJoin('vc.texts', 'fText');
				}

				$qb->andWhere('fText.h1 LIKE :h1')
					->setParameter('h1', "%$value%");
			});
		$grid->addColumnText('group', $columnPrefix . 'groups')->setRenderer(static function(VirtualCategory $row) {
			$html = Html::el();

			foreach ($row->inGroups as $g) {
				$html->addHtml(Html::el('div')->setText($g->group->getText()->title));
			}

			return $html;
		});
		$grid->addColumnText('isLocked', $columnPrefix . 'isLocked')
			->setRenderer(static function(VirtualCategory $row) {
				if ($row->isLocked()) {
					return Html::el('span class="btn btn-success"')
						->addHtml(Html::el('i class="fa fa-check"'));
				}

				return '';
			})->setAlign('center')
			->setFilterSelect([
				'' => '',
				0  => $this->t('default.no'),
				1  => $this->t('default.yes'),
			]);

		// Filter
		$grid->addFilterSelect('group', $columnPrefix . 'group', ['' => ''] + $this->virtualCategoryGroups->getOptionsForSelect())
			->setCondition(function(QueryBuilder $qb, $value) {
				$qb->andWhere('inG.group = :group')->setParameter('group', $value);
			});

		// Action
		$grid->addAction('edit', '', 'edit!', ['id' => 'id'])
			->setIcon('edit')
			->setBsType('primary')
			->addClass('ajax');
		if (EshopAdvancedFeatureConfig::load('allowRelated')) {
			$grid->addAction('related', '', 'related!')->setIcon('link')->setBsType('default')
				->addClass('ajax');
		}
		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		$grid->addGroupAction('default.delete')->onSelect[] = [$this, 'handleDelete'];

		// Prototype
		$grid->getColumn('group')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('isLocked')->getElementPrototype('th')->addClass('w1nw');

		return $grid;
	}

	protected function createComponentForm(): VirtualCategoryForm
	{
		$control = $this->formFactory->create($this->siteIdent);

		if ($this->getParameter('id')) {
			$control->setVirtualCategory((int) $this->getParameter('id'));
		}

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[] = function(BaseForm $baseForm) {
				$this->getPresenter()->redrawControl('flashes');
				$this['grid']->reload();
				$this->handleEdit((int) $baseForm->getCustomData('virtualCategoryId'));
			};

			$control['form']->onSuccessSaveAndClose[] = function() {
				$this->getPresenter()->payload->hideModal = true;
				$this->getPresenter()->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']['saveControl']->closeModalOnCancel();
		};

		return $control;
	}

	protected function createComponentRelatedForm(): Multiplier
	{
		return new Multiplier(function($id): RelatedForm {
			return $this->relatedFormFactory->create((int) $id);
		});
	}

	//		protected function createComponentNavigation(): Navigation
	//		{
	//			return $this->navigationFactory->create();
	//		}
}
