<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\AdminModule\Components\VirtualCategory;

use Core\AdminModule\Model\Sites;
use Core\Model\Helpers\FormHelper;
use Core\Model\Helpers\Strings;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategories;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategoryGroups;
use EshopAdvancedFeature\Model\Entities\VirtualCategory;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryInGroup;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryText;
use EshopCatalog\AdminModule\Model\Categories;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeatureValues;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\FrontModule\Model\CacheService;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use EshopCatalog\FrontModule\Model\FilterService as FrontFilterService;

class VirtualCategoryForm extends BaseControl
{
	/** @var int|null @persistent */
	public ?int $id = null;

	public string           $siteIdent;
	public ?VirtualCategory $virtualCategory = null;

	protected VirtualCategories      $virtualCategories;
	protected Features               $features;
	protected FeatureValues          $featureValues;
	protected CacheService           $cacheService;
	protected Manufacturers          $manufacturers;
	protected Categories             $categories;
	protected Sites                  $sites;
	protected VirtualCategoryGroups  $groupService;
	protected FrontFilterService     $frontFilterService;

	public function __construct(
		string                 $siteIdent,
		VirtualCategories      $virtualCategories,
		CacheService           $cacheService,
		Sites                  $sites,
		Features               $features,
		FeatureValues          $featureValues,
		Manufacturers          $manufacturers,
		Categories             $categories,
		VirtualCategoryGroups  $groupService,
		FrontFilterService     $frontFilterService
	)
	{
		$this->siteIdent              = $siteIdent;
		$this->virtualCategories      = $virtualCategories;
		$this->features               = $features;
		$this->featureValues          = $featureValues;
		$this->cacheService           = $cacheService;
		$this->manufacturers          = $manufacturers;
		$this->categories             = $categories;
		$this->sites                  = $sites;
		$this->groupService           = $groupService;
		$this->frontFilterService     = $frontFilterService;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();

		$titlePrefix = 'eshopAdvancedFeature.virtualCategory.';
		$form->addText('url', $titlePrefix . 'url')
			->setIsMultilanguage();
		$form->addFilesManager('icon', $titlePrefix . 'image');
		$form->addText('h1', $titlePrefix . 'h1')
			->setIsMultilanguage();
		$form->addEditor('description', $titlePrefix . 'shortDescription')
			->setIsMultilanguage()
			->setDisableAutoP(false)
			->setHeight(100);
		$form->addEditor('longDescription', $titlePrefix . 'longDescription')
			->setIsMultilanguage()
			->setDisableAutoP(false)
			->setHeight(100);
		$form->addText('filteringTitle', $titlePrefix . 'filteringTitle')
			->setIsMultilanguage();
		$form->addText('menuTitle', $titlePrefix . 'menuTitle')
			->setIsMultilanguage();

		$form->addText('pageTitle', $titlePrefix . 'pageTitle')
			->setIsMultilanguage();
		$form->addTextArea('pageDescription', $titlePrefix . 'pageDescription')
			->setIsMultilanguage();

		$form->addCheckboxList('groups', $this->t($titlePrefix . 'groups'), $this->groupService->getOptionsForSelect())
			->setTranslator(null);
		$form->addCheckboxNestedList('categories', $this->t($titlePrefix . 'categories'), $this->categories->getFlatTree($this->siteIdent))
			->setTranslator(null);
		$form->addCheckboxList('manufacturers', $this->t($titlePrefix . 'manufacturers'), $this->manufacturers->getOptionsForSelect())
			->setTranslator(null);

		// Features
		$flat         = [];
		$featureTexts = $this->features->getOptionsForSelect();
		foreach ($this->featureValues->getOptionsForSelect() as $featureId => $values) {
			$flat[] = [
				'id'     => 'f' . $featureId,
				'name'   => $featureTexts[$featureId],
				'parent' => 0,
			];

			foreach ($values as $valueId => $value) {
				$flat[] = [
					'id'     => $valueId,
					'name'   => $value,
					'parent' => 'f' . $featureId,
				];
			}
		}
		$form->addCheckboxNestedList('featureValues', $titlePrefix . 'featureValues', $flat);

		$form->addSaveCancelControl();

		$form['saveControl']['save']->setValidationScope([]);
		$form['saveControl']['saveAndClose']->setValidationScope([]);

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$this->em->beginTransaction();
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var VirtualCategoryText[] $texts */
			$texts = [];

			if ($this->id) {
				$virtualCategory = $this->virtualCategories->get($this->id);
				$texts           = $virtualCategory ? $virtualCategory->texts->toArray() : [];
			} else {
				$virtualCategory = new VirtualCategory($this->siteIdent);
			}

			foreach ($langValues as $l => $v) {
				if ($v['url'] && !Strings::startsWith($v['url'], '/')) {
					$v['url'] = '/' . $v['url'];
				}

				if (!isset($texts[$l]) && $v['url']) {
					$texts[$l] = new VirtualCategoryText($virtualCategory, $l, $v['url']);
				}

				if (!$v['url']) {
					if ($texts[$l] instanceof VirtualCategoryText) {
						$this->em->remove($texts[$l]);
					}

					unset($texts[$l]);
					continue;
				}

				$texts[$l]->setUrl($v['url']);
				foreach ([
					         'h1', 'description', 'longDescription',
					         'menuTitle', 'pageTitle', 'pageDescription',
				         ] as $col) {
					$texts[$l]->{$col} = $v[$col];
				}

				$this->em->persist($texts[$l]);
				$virtualCategory->texts->set($l, $texts[$l]);
			}

			// Groups
			$formVals   = array_flip($values->groups);
			$entityVals = $virtualCategory->inGroups->toArray();
			foreach (array_diff_key($formVals, $entityVals) as $key => $value) {
				$entity = new VirtualCategoryInGroup(
					$this->groupService->getReference($key),
					$virtualCategory,
				);

				$this->em->persist($entity);
			}

			foreach (array_diff_key($entityVals, $formVals) as $value => $inGroup) {
				$virtualCategory->inGroups->removeElement($inGroup);
				$this->em->remove($inGroup);
			}

			// Categories
			$formVals   = array_flip($values->categories);
			$entityVals = $virtualCategory->categories->toArray();
			foreach (array_diff_key($formVals, $entityVals) as $key => $value) {
				$virtualCategory->categories->add($this->categories->getReference((int) $key));
			}

			foreach (array_diff_key($entityVals, $formVals) as $value) {
				$virtualCategory->categories->removeElement($value);
			}

			// Manufacturers
			$formVals   = array_flip($values->manufacturers);
			$entityVals = $virtualCategory->manufacturers->toArray();
			foreach (array_diff_key($formVals, $entityVals) as $key => $value) {
				$virtualCategory->manufacturers->add($this->manufacturers->getReference((int) $key));
			}

			foreach (array_diff_key($entityVals, $formVals) as $value) {
				$virtualCategory->manufacturers->removeElement($value);
			}

			// Features
			$formVals = [];
			foreach ($values->featureValues as $value) {
				if (Strings::startsWith($value, 'f')) {
					continue;
				}

				$formVals[$value] = $value;
			}
			$entityVals = $virtualCategory->featureValues->toArray();
			foreach (array_diff_key($formVals, $entityVals) as $key => $value) {
				$virtualCategory->featureValues->add($this->featureValues->getReference((int) $key));
			}

			foreach (array_diff_key($entityVals, $formVals) as $value) {
				$virtualCategory->featureValues->removeElement($value);
			}

			$virtualCategory->icon = $values->icon;
			$virtualCategory->lock();

			$this->em->persist($virtualCategory);
			$this->em->flush();
			$this->em->commit();

			$form->addCustomData('virtualCategoryId', $virtualCategory->getId());
			$this->getPresenter()->flashMessageSuccess('default.saved');
		} catch (\Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}

			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function getVirtualCategory(): ?VirtualCategory
	{
		if (!$this->virtualCategory && $this->id) {
			$this->virtualCategory = $this->virtualCategories->get($this->id);
		}

		return $this->virtualCategory;
	}

	public function setVirtualCategory(int $id)
	{
		$this->id = $id;
		$vc       = $this->virtualCategories->get($id);

		if (!$vc) {
			throw new InvalidArgumentException('');
		}

		$this->virtualCategory = $vc;

		$d = [
			'ico'           => $vc->icon,
			'groups'        => $vc->inGroups->getKeys(),
			'featureValues' => $vc->featureValues->getKeys(),
			'manufacturers' => $vc->manufacturers->getKeys(),
			'categories'    => $vc->categories->getKeys(),
		];

		foreach ($vc->texts as $lang => $text) {
			foreach ([
				         'h1', 'description', 'longDescription',
				         'menuTitle', 'pageTitle', 'pageDescription',
			         ] as $v) {
				$d[$v][$lang] = $text->{$v};
			}

			$d['url'][$lang] = $text->getUrl();
		}

		$this['form']->setDefaults($d);
	}
}
