<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\FrontModule\Model\Subscribers;

use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Arrays;
use EshopAdvancedFeature\Model\Helpers\VirtualCategoryHelper;
use EshopAdvancedFeature\Model\Navigation\VirtualCategoryGroup;
use EshopAdvancedFeature\Model\VirtualCategories as DefaultVirtualCategories;
use EshopCatalog\FrontModule\Components\ProductsFilter;
use EshopCatalog\FrontModule\Model\Dao\Category;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\Event;
use Core\Model\Helpers\Strings;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Components\Navigation;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Event\FilterLinkEvent;
use EshopCatalog\FrontModule\Presenters\DefaultPresenter;
use EshopCatalog\Model\Config;
use Nette\Http\Request;
use Nette\Http\Url;
use Nette\Localization\Translator;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Core\Model\Entities\EntityManagerDecorator;
use EshopAdvancedFeature\Model\PrefixSuffixGenerator;
use EshopCatalog\FrontModule\Model\Event\CategoryHeadEvent;

class VirtualCategoriesSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator   $entityManager;
	protected Translator               $translator;
	protected EventDispatcher          $eventDispatcher;
	protected PrefixSuffixGenerator    $prefixSuffixGenerator;
	protected Categories               $categories;
	protected DefaultVirtualCategories $defaultVirtualCategories;
	protected Request                  $request;

	protected array $cParentPath = [];

	public function __construct(
		EntityManagerDecorator   $entityManager,
		Translator               $translator,
		EventDispatcher          $eventDispatcher,
		PrefixSuffixGenerator    $prefixSuffixGenerator,
		Categories               $categories,
		DefaultVirtualCategories $defaultVirtualCategories,
		Request                  $request
	)
	{
		$this->entityManager            = $entityManager;
		$this->translator               = $translator;
		$this->eventDispatcher          = $eventDispatcher;
		$this->prefixSuffixGenerator    = $prefixSuffixGenerator;
		$this->categories               = $categories;
		$this->defaultVirtualCategories = $defaultVirtualCategories;
		$this->request                  = $request;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopCatalog.default.actionCategory.head'    => ['actionCategoryHead', 100],
			ProductsFilter::class . '::createFilterLink'  => 'createFilterLink',
			ProductsFilter::class . '::createSortLink'    => 'createSortLink',
			Navigation::class . '::onAttach'              => 'navigationOnAttach',
			Categories::class . '::loadRelated'           => 'loadRelated',
			ProductsFilter::class . '::beforeLoadFilters' => 'beforeLoadFilters',
		];
	}

	public function actionCategoryHead(CategoryHeadEvent $event): void
	{
		$filters = $event->activeFilters;

		$data = $this->prefixSuffixGenerator->generate(
			$filters['feature'] ?? [],
			$filters['manu'] ?? [],
			$filters['tags'] ?? [],
			$filters['priceRange'] ?? [],
			$filters['range'] ?? [],
			$event->category->id,
		);

		$activeNav    = $event->presenter->getActiveNavigation();
		$allowRelated = Config::load('allowRelatedCategories', false);
		if ($allowRelated && $activeNav && $activeNav->getParam('virtualCategoryData')) {
			$virtualCategory = $activeNav->getParam('virtualCategoryData');
			$related         = [];

			if (isset($virtualCategory['related'])) {
				$event2 = new Event([
					'related' => &$virtualCategory['related'],
					'lang'    => $activeNav->lang,
				]);

				$this->eventDispatcher->dispatch($event2, Categories::class . '::loadRelated');
				foreach ($virtualCategory['related'] as $k => $row) {
					if ($row['targetKey'] === 'category') {
						$cat = $this->categories->get($row['targetId']);
						if ($cat)
							$row['targetEntity'] = $cat;
					}

					if (isset($row['targetEntity']))
						$related[] = $row['targetEntity'];
				}
			}

			$event->category->related = $related;
		}

		$vcTexts = $activeNav->getParam('virtualCategoryData') ?? [];

		$event->categoryName = $vcTexts['h1']
			?: $this->prefixSuffixGenerator->addPrefixSuffix($event->categoryName, $data['prefix'], $data['suffix']);

		if ($vcTexts['description']) {
			$event->description = $vcTexts['description'];
		}

		$event->title = $vcTexts['page_title']
			?: $this->prefixSuffixGenerator->addPrefixSuffix($event->title, $data['prefix'], $data['suffix']);

		if ($vcTexts['page_description']) {
			$event->metaDescription = $vcTexts['page_description'];
		}
	}

	public function createFilterLink(FilterLinkEvent $event): void
	{
		if (empty($event->categories) || count($event->categories) > 1) {
			return;
		}

		$filterFeature     = $event->currentFilter['feature'] ?? [];
		$filtersManu       = $event->currentFilter['manu'] ?? [];
		$filtersTags       = $event->currentFilter['tag'] ?? [];
		$filtersPriceRange = $event->currentFilter['priceRange'] ?? [];
		$filtersRange      = $event->currentFilter['range'] ?? [];

		if ($filtersTags || $filtersPriceRange || $filtersRange) {
			return;
		}

		$createUrl  = true;
		$categoryId = $event->categories[0];
		$category   = $this->categories->get($categoryId);

		switch ($event->type) {
			case 'feature':
				if (isset($filterFeature[$event->key]) && Arrays::contains($filterFeature[$event->key], $event->value)) {
					$key = array_search($event->value, $filterFeature[$event->key], true);
					unset($filterFeature[$event->key][$key]);

					if (count($filterFeature[$event->key]) === 0) {
						unset($filterFeature[$event->key]);
					}
				} else {
					$filterFeature[$event->key][] = $event->value;

					if (count($filterFeature[$event->key]) > 1) {
						$createUrl = false;
					}
				}

				break;
			case 'manu':
				if (Arrays::contains($filtersManu, $event->value)) {
					$key = array_search($event->value, $filtersManu, true);
					unset($filtersManu[$key]);
				} else {
					$filtersManu[] = $event->value;

					if (count($filtersManu) > 1) {
						$createUrl = false;
					}
				}
				break;
			case 'tag':
				$createUrl = false;
				break;
		}

		if (!$category || count($filtersManu) > 1) {
			$createUrl = false;
		}

		if (empty($filtersManu) && empty($filterFeature)) {
			$createUrl = false;
		}

		foreach ($filterFeature as $k => $vals) {
			if (count($vals) > 1) {
				$createUrl = false;
			}
		}

		$parentPath    = $this->getParentPath($category);
		$event->result = ($parentPath ? '/' . $this->getParentPath($category) : '');

		if ($createUrl) {
			$data = $this->prefixSuffixGenerator->generate(
				$filterFeature,
				$filtersManu,
				$filtersTags,
				$filtersPriceRange,
				$filtersRange,
				$categoryId,
			);

			$event->result .= '/' . Strings::webalize($this->prefixSuffixGenerator->addPrefixSuffix($category->getNameH1(), $data['prefix'], $data['suffix']));

			if ($event->result !== rtrim($this->request->getUrl()->getPath(), '/')) {
				VirtualCategoryHelper::prepareCreateVirtualCategory($event->result, [$categoryId], array_unique(array_merge(...$filterFeature)), $filtersManu);
			}
		} else {
			$event->result = null;
		}
	}

	public function createSortLink(FilterLinkEvent $event): void
	{
		if (empty($event->currentFilter)) {
			return;
		}

		$url = new Url($this->request->getUrl());
		if ($event->value && $event->value !== 'recommended') {
			$url->setQueryParameter('productsFilter-sort', urlencode($event->value));
		} else {
			$url->setQueryParameter('productsFilter-sort', null);
		}

		$event->result = $url->getAbsoluteUrl();
	}

	protected function getParentPath(Category $category): string
	{
		$key = $category->getId() . '_' . $this->translator->getLocale();

		if (!isset($this->cParentPath[$key])) {
			$this->cParentPath[$key] = implode('/', array_map(
					static fn(Category $c) => $c->getAttr('originAlias') ?: $c->alias,
					array_reverse($category->getParentPath())
				)
			);
		}

		return $this->cParentPath[$key];
	}

	public function navigationOnAttach(ControlEvent $event): void
	{
		/** @var Navigation $control */
		$control     = $event->control;
		$groupsInNav = [];

		foreach ($control->getNavs() as $nav) {
			if ($nav->componentType !== VirtualCategoryGroup::COMPONENT_TYPE) {
				continue;
			}

			$nav->componentParams['category']            = 'virtualCategory_' . $nav->componentParams['group'];
			$groupsInNav[$nav->componentParams['group']] = $nav;
		}

		foreach ($this->defaultVirtualCategories->findNavigationsByGroups(array_keys($groupsInNav)) as $groupId => $navs) {
			$control->otherCategories['virtualCategory_' . $groupId] = $this->defaultVirtualCategories->createCategoryDaoFromArray($navs, $groupsInNav[$groupId]);
		}
	}

	public function loadRelated(Event $event): void
	{
		$ids = [];

		foreach ($event->data['related'] as $k => $rows) {
			if (isset($rows['targetKey'])) {
				if ($rows['targetKey'] === 'virtualCategory') {
					$ids[$rows['targetId']] = [
						'k1' => $k,
					];
				}
			} else {
				foreach ($rows as $k2 => $row) {
					if ($row['targetKey'] !== 'virtualCategory') {
						continue;
					}

					$ids[$row['targetId']] = [
						'k1' => $k,
						'k2' => $k2,
					];
				}
			}
		}

		if ($ids) {
			foreach ($this->defaultVirtualCategories->getEr()->createQueryBuilder('vc')
				         ->select('vc.id, vct.url, vct.h1, vc.icon, vct.description, vct.longDescription')
				         ->innerJoin('vc.texts', 'vct', Join::WITH, 'vct.locale = :locale')
				         ->where('vc.id IN (:ids)')
				         ->setParameters([
					         'locale' => $event->data['lang'],
					         'ids'    => array_keys($ids),
				         ])->getQuery()->getArrayResult() as $row) {
				if (!$row['url']) {
					continue;
				}

				$cat                   = new Category();
				$cat->name             = $row['h1'];
				$cat->nameH1           = $row['h1'];
				$cat->alias            = $cat->name ? Strings::webalize($cat->name) : '';
				$cat->link             = $row['url'];
				$cat->image            = $row['icon'];
				$cat->shortDescription = $row['description'];
				$cat->description      = $row['longDescription'];

				$k = $ids[$row['id']];
				if (isset($k['k2'])) {
					$event->data['related'][$k['k1']][$k['k2']]['targetEntity'] = $cat;
				} else {
					$event->data['related'][$k['k1']]['targetEntity'] = $cat;
				}
			}
		}
	}

	public function beforeLoadFilters(ControlEvent $event): void
	{
		/** @var ProductsFilter $control */
		$control = $event->control;
		/** @var DefaultPresenter $presenter */
		$presenter = $control->getPresenter(false);

		if (!$presenter)
			return;

		$nav          = $presenter->getActiveNavigation();
		$groupFilters = $nav->componentParams['virtualCategoryData']['group']['filters'] ?? null;

		if ($groupFilters) {
			$presenter['productsFilter']->allowedFeatures = $groupFilters;
		}
	}
}
