<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model\Entities;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;
use Nette\Utils\DateTime;

/**
 * @ORM\Table("eshop_advanced_feature__feature_navigation", indexes={@ORM\Index(name="eaffn_site", columns={"site_ident"})})
 * @ORM\Entity
 */
class FeatureNavigation
{
	use TId;

	/**
	 * @var FeatureNavigationText[]|ArrayCollection
	 * @ORM\OneToMany(targetEntity="FeatureNavigationText", mappedBy="featureNavigation", indexBy="locale")
	 */
	protected $texts;

	/**
	 * @var int
	 * @ORM\Column(name="navigation_id", type="integer", nullable=true)
	 */
	public $navigationId;

	/**
	 * @var bool
	 * @ORM\Column(name="dynamic_params", type="boolean", nullable=false, options={"default": 0})
	 */
	public $dynamicParams;

	/**
	 * @var array
	 * @ORM\Column(name="params", type="array", nullable=false)
	 */
	public $params;

	/**
	 * @var int
	 * @ORM\Column(name="is_locked", type="smallint", nullable=false, options={"unsigned": true, "default": 0})
	 */
	protected int $isLocked = 0;

	/**
	 * @var string|null
	 * @ORM\Column(name="ico", type="string", length=255, nullable=true)
	 */
	public ?string $ico = null;

	/**
	 * @ORM\Column(name="site_ident", type="string", nullable=true)
	 */
	public ?string $siteIdent = null;

	/**
	 * @var ArrayCollection|FeatureNavigationInGroup[]
	 * @ORM\OneToMany(targetEntity="FeatureNavigationInGroup", mappedBy="featureNavigation", cascade={"all"}, orphanRemoval=true)
	 */
	protected $groups;

	/**
	 * @var ArrayCollection|FeatureNavigationRelated[]
	 * @ORM\OneToMany(targetEntity="FeatureNavigationRelated", mappedBy="featureNavigation")
	 */
	protected $related;

	/**
	 * @ORM\Column(name="created", type="datetime", options={"default": "CURRENT_TIMESTAMP"})
	 */
	public ?\DateTimeInterface $created = null;

	public function __construct(?int $navigationId = null, array $params)
	{
		$this->navigationId  = $navigationId;
		$this->params        = $params;
		$this->dynamicParams = false;
		$this->texts         = new ArrayCollection();
		$this->groups        = new ArrayCollection();
		$this->related       = new ArrayCollection();
		$this->created       = new DateTime('now');
	}

	public function lock(bool $lock = true): self
	{
		$this->isLocked = $lock ? 1 : 0;

		return $this;
	}

	public function isLocked(): bool { return (bool) $this->isLocked; }

	public function getTexts() { return $this->texts; }

	public function getText(?string $locale = null) { return $locale ? $this->texts[$locale] ?? null : $this->texts->first(); }

	public function setTexts(array $texts)
	{
		$this->texts = new ArrayCollection($texts);
	}

	public function getGroups() { return $this->groups; }

	public function setGroups(array $groups)
	{
		$this->groups = new ArrayCollection($groups);
	}
}
