<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model\Entities;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\Manufacturer;
use Nette\Utils\DateTime;

/**
 * @ORM\Table("eshop_advanced_feature__virtual_category", indexes={@ORM\Index(name="eafvc_site", columns={"site_ident"})})
 * @ORM\Entity
 */
class VirtualCategory
{
	use TId;

	/**
	 * @var ArrayCollection<VirtualCategoryText>
	 * @ORM\OneToMany(targetEntity="VirtualCategoryText", mappedBy="virtualCategory", indexBy="locale")
	 */
	public Collection $texts;

	/**
	 * @ORM\Column(name="is_locked", type="smallint", nullable=false, options={"unsigned": true, "default": 0})
	 */
	protected int $isLocked = 0;

	/**
	 * @ORM\Column(name="icon", type="string", length=255, nullable=true)
	 */
	public ?string $icon = null;

	/**
	 * @ORM\Column(name="site_ident", type="string", nullable=false)
	 */
	public string $siteIdent;

	/**
	 * @var Collection<Category>
	 * @ORM\ManyToMany(targetEntity="EshopCatalog\Model\Entities\Category", indexBy="id")
	 * @ORM\JoinTable(name="eshop_advanced_feature__virtual_category_categories",
	 *     joinColumns={@ORM\JoinColumn(name="virtual_category_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="category_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	public Collection $categories;

	/**
	 * @var Collection<Manufacturer>
	 * @ORM\ManyToMany(targetEntity="EshopCatalog\Model\Entities\Manufacturer", indexBy="id")
	 * @ORM\JoinTable(name="eshop_advanced_feature__virtual_category_manufacturers",
	 *     joinColumns={@ORM\JoinColumn(name="virtual_category_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="manufacturer_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	public Collection $manufacturers;

	/**
	 * @var Collection<FeatureValue>
	 * @ORM\ManyToMany(targetEntity="EshopCatalog\Model\Entities\FeatureValue", indexBy="id")
	 * @ORM\JoinTable(name="eshop_advanced_feature__virtual_category_features",
	 *     joinColumns={@ORM\JoinColumn(name="virtual_category_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="feature_value_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	public Collection $featureValues;

	/**
	 * @ORM\Column(name="created", type="datetime", options={"default": "CURRENT_TIMESTAMP"})
	 */
	public \DateTimeInterface $created;

	/**
	 * @var ArrayCollection<VirtualCategoryRelated>
	 * @ORM\OneToMany(targetEntity="VirtualCategoryRelated", mappedBy="virtualCategory")
	 */
	public Collection $related;

	/**
	 * @var ArrayCollection<VirtualCategoryInGroup>
	 * @ORM\OneToMany(targetEntity="VirtualCategoryInGroup", mappedBy="virtualCategory", indexBy="group_id")
	 */
	public Collection $inGroups;

	public function __construct(
		string $siteIdent
	)
	{
		$this->siteIdent     = $siteIdent;
		$this->created       = new DateTime();
		$this->texts         = new ArrayCollection();
		$this->categories    = new ArrayCollection();
		$this->manufacturers = new ArrayCollection();
		$this->featureValues = new ArrayCollection();
		$this->related       = new ArrayCollection();
		$this->inGroups      = new ArrayCollection();
	}

	public function lock(bool $lock = true): void { $this->isLocked = $lock ? 1 : 0; }

	public function isLocked(): bool { return (bool) $this->isLocked; }

	public function getText(?string $locale = null): ?VirtualCategoryText { return $locale ? $this->texts[$locale] ?? null : $this->texts->first(); }
}
