<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model\Navigation;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategories;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategoryGroups;
use EshopAdvancedFeature\Model\FeatureNavigationGroups;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Navigations\Model\Providers\INavigationItem;
use Nette\Utils\ArrayHash;
use Pages\AdminModule\Model\TemplatePages;
use EshopAdvancedFeature\Model\VirtualCategories as FrontVirtualCategories;

class VirtualCategoryGroup extends BaseNavigation implements INavigationItem
{
	public const COMPONENT_TYPE = 'eshopAdvancedFeature.navigation.virtualCategoryGroup';

	protected $title = 'group';

	protected $presenter = 'EshopAdvancedFeature:Front:Default';
	protected $action    = 'virtualCategoryGroup';

	protected VirtualCategoryGroups   $groupService;
	protected VirtualCategories       $virtualCategories;
	protected FeatureNavigationGroups $featureNavigationGroups;
	protected FrontVirtualCategories  $frontVirtualCategories;

	protected TemplatePages $templatePagesService;

	public function __construct(
		VirtualCategoryGroups  $groupService,
		VirtualCategories      $virtualCategories,
		TemplatePages          $templatePages,
		FrontVirtualCategories $frontVirtualCategories
	)
	{
		$this->groupService           = $groupService;
		$this->virtualCategories      = $virtualCategories;
		$this->templatePagesService   = $templatePages;
		$this->frontVirtualCategories = $frontVirtualCategories;
	}

	public function getFormContainer(BaseForm $baseForm): BaseContainer
	{
		$container = new BaseContainer();

		$container->addSelect('group', $this->t('eshopAdvancedFeature.navigationBuilder.group'), $this->groupService->getOptionsForSelect())
			->setTranslator(null);

		$container->addSelect('templatePage', $this->t('eshopAdvancedFeature.navigationBuilder.template'), $this->templatePagesService->getOptionsForSelectGrouped())
			->setTranslator(null);

		return $container;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		return true;
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action): bool
	{
		return $presenter === $this->presenter && $action === $this->action;
	}

	/**
	 * @param DaoNavigationItem $navigation
	 * @param array             $urlParams
	 *
	 * @return array|mixed
	 */
	public function routerIn($navigation, $urlParams = []): ?array
	{
		return [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => $navigation->componentParams['group'],
		];
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		return '/' . $navigation->alias;
	}

	/**
	 * @param string                 $lang
	 * @param DaoNavigationItem|null $navigation
	 * @param bool                   $siteMapData
	 *
	 * @return array
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$arr = [];

		$group = $this->featureNavigationGroups->getGroup((int) $navigation->componentParams['group']);
		$navs  = $this->frontVirtualCategories->findNavigationsByGroups([$group['id']]);

		if (!is_array($navs[$group['id']])) {
			return [];
		}

		$category = $this->frontVirtualCategories->createCategoryDaoFromArray($navs[$group['id']], $navigation);
		if ($category) {
			foreach ($category->getChild() as $child) {
				$data = [
					'linkPath' => $child->link,
					'locale'   => $lang,
				];

				if ($siteMapData) {
					$siteMapUrl = new SiteMapUrl();
					$seo        = [
						'robots'       => 'index, follow',
						'addToSiteMap' => 1,
					];
					$siteMapUrl->setSeo($seo, null, true);
					$data['siteMapData']['default'] = $siteMapUrl;
				}
				$arr[] = $data;
			}
		}

		return $arr;
	}
}
