<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model\Subscribers;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Sites;
use EshopAdvancedFeature\Model\VirtualCategories;
use EshopCatalog\Model\Navigation\Home;
use Nette\Localization\Translator;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Core\Model\Entities\EntityManagerDecorator;
use Navigations\Model\Event\RouteInEvent;
use Navigations\Model\Event\RouteOutEvent;
use EshopCatalog\Model\Navigation\BaseNavigation;
use Navigations\Model\Navigations;

class VirtualCategoriesNavigationSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected VirtualCategories      $virtualCategories;
	protected Navigations            $navigations;
	protected Sites                  $sites;
	protected Translator             $translator;

	public function __construct(
		EntityManagerDecorator $em,
		VirtualCategories      $virtualCategories,
		Navigations            $navigations,
		Sites                  $sites,
		Translator             $translator
	)
	{
		$this->em                = $em;
		$this->virtualCategories = $virtualCategories;
		$this->navigations       = $navigations;
		$this->sites             = $sites;
		$this->translator        = $translator;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			Navigations::class . '::routeInNotFoundPreRedirect' => ['routeInNotFound', 110],
			BaseNavigation::class . '::routeOut'                => ['routeOut', 110],
		];
	}

	public function routeInNotFound(RouteInEvent $event): void
	{
		$url    = '/' . ltrim($event->urlParams['path'], '/');
		$locale = $event->urlParams['locale'];

		$data = $this->virtualCategories->getNavigationDataByUrl($url, $locale);

		if ($data) {
			$filter = [];
			$return = [
				'presenter' => 'EshopCatalog:Front:Default',
				'action'    => 'category',
			];

			if ($data['cats']) {
				$return += [
					'id' => is_array($data['cats']) ? implode('|', $data['cats']) : (int) $data['cats'],
				];
			}

			if ($data['manu']) {
				$filter['fm'] = $data['manu'];
			}

			if ($data['features']) {
				$filter['ff'] = $data['features'];
			}

			if (!empty($filter)) {
				$return += [
					'productsFilter-filter' => $filter,
				];
			}

			$activeNavs = $this->navigations->getPublishedByComponent(Home::COMPONENT_NAME, $locale);
			if ($activeNavs) {
				/** @var DaoNavigationItem $activeNav */
				$activeNav = array_shift($activeNavs);

				$activeNav->setParam('virtualCategory', $data['id']);
				$activeNav->setParam('virtualCategoryData', $data);
				$activeNav->setParam('url', $url);
				$activeNav->link = $url;

				$return['activeNavigation'] = $activeNav;
			}

			$event->return = $return;
			$event->stopPropagation();
		}
	}

	public function routeOut(RouteOutEvent $event): void
	{
		$activeNav = $event->navigation;

		if ($activeNav->getParam('virtualCategory')) {
			unset($event->urlParams['productsFilter-filter']);

			if ($event->urlParams['productsFilter-fr'] === null) {
				unset($event->urlParams['productsFilter-fr']);
			}
		}
	}
}
