<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Helpers\Strings;
use Core\Model\Sites;
use EshopAdvancedFeature\Model\Entities\VirtualCategory;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryRelated;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryText;
use EshopAdvancedFeature\Model\Helpers\VirtualCategoryHelper;
use EshopCatalog\FrontModule\Model\Dao\Category;
use Core\Model\Helpers\BaseFrontEntityService;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\Features;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\Manufacturer;

class VirtualCategories extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = VirtualCategory::class;

	protected Sites    $sites;
	protected Features $features;

	protected array  $prepareCreateVirtualCategory = [];
	protected ?array $cRelated                     = null;

	public function __construct(
		Sites    $sites,
		Features $features
	)
	{
		$this->sites    = $sites;
		$this->features = $features;
	}

	public function getNavigationDataByUrl(string $url, ?string $locale = null): array
	{
		return $this->getNavigationDataByUrlMultiple([$url], $locale)[$url] ?? [];
	}

	public function getNavigationDataByRelations(array $data): array
	{
		$conn   = $this->em->getConnection();
		$result = [];

		foreach ($data as $url => $row) {
			$where = ["vc.site_ident = '" . $this->sites->getCurrentSite()->getIdent() . "'"];

			$query = "SELECT vc.id FROM eshop_advanced_feature__virtual_category vc ";

			if ($row['categories']) {
				$count   = count($row['categories']);
				$where[] = "EXISTS (
					SELECT COUNT(cat.category_id) as catCount
					FROM eshop_advanced_feature__virtual_category_categories cat 
					WHERE cat.virtual_category_id = vc.id
					HAVING {$count} = SUM(
						CASE WHEN cat.category_id IN (" . implode(',', $row['categories']) . ") THEN 1 ELSE 0 END
					) AND {$count} = catCount
				)";
			} else {
				$where[] = "NOT EXISTS (
					SELECT 1
					FROM eshop_advanced_feature__virtual_category_categories cat
					WHERE cat.virtual_category_id = vc.id
				)";
			}

			if ($row['manufacturers']) {
				$count   = count($row['manufacturers']);
				$where[] = "EXISTS (
					SELECT COUNT(manu.manufacturer_id) as manuCount
					FROM eshop_advanced_feature__virtual_category_manufacturers manu 
					WHERE manu.virtual_category_id = vc.id
					HAVING {$count} = SUM(
						CASE WHEN manu.manufacturer_id IN (" . implode(',', $row['manufacturers']) . ") THEN 1 ELSE 0 END
					) AND {$count} = manuCount
				)";
			} else {
				$where[] = "NOT EXISTS (
					SELECT 1
					FROM eshop_advanced_feature__virtual_category_manufacturers manu
					WHERE manu.virtual_category_id = vc.id
				)";
			}

			if ($row['featureValues']) {
				$count   = count($row['featureValues']);
				$where[] = "EXISTS (
					SELECT COUNT(f.feature_value_id) as fCount
					FROM eshop_advanced_feature__virtual_category_features f 
					WHERE f.virtual_category_id = vc.id
					HAVING {$count} = SUM(
						CASE WHEN f.feature_value_id IN (" . implode(',', $row['featureValues']) . ") THEN 1 ELSE 0 END
					) AND {$count} = fCount
				)";
			} else {
				$where[] = "NOT EXISTS (
					SELECT 1
					FROM eshop_advanced_feature__virtual_category_features f
					WHERE f.virtual_category_id = vc.id
				)";
			}

			$data = $conn->fetchAssociative($query . 'WHERE ' . implode(' AND ', $where));

			if ($data) {
				$result[$url] = $data['id'];
			}
		}

		return $result;
	}

	public function getNavigationsIdByUrlMultiple(array $urls, ?string $locale = null): array
	{
		$locale = $locale ?: $this->translator->getLocale();

		$urlQuery = count($urls) === 1
			? '= \'' . $urls[0] . '\''
			: 'IN (\'' . implode('\',\'', $urls) . '\')';

		$conn   = $this->em->getConnection();
		$result = [];

		foreach ($conn->fetchAllAssociative("SELECT vct.id, vct.url
		FROM eshop_advanced_feature__virtual_category_text vct
		INNER JOIN eshop_advanced_feature__virtual_category vc ON vct.id = vc.id AND vc.site_ident = :siteIdent
		WHERE vct.locale = :locale AND vct.url {$urlQuery}", [
			'locale'    => $locale,
			'siteIdent' => $this->sites->getCurrentSite()->getIdent(),
		]) as $virtualCategory) {
			$result[$virtualCategory['url']] = $virtualCategory['id'];
		}

		return $result;
	}

	public function getNavigationDataByUrlMultiple(array $urls, ?string $locale = null): array
	{
		$locale = $locale ?: $this->translator->getLocale();

		$urlQuery = count($urls) === 1
			? '= \'' . $urls[0] . '\''
			: 'IN (\'' . implode('\',\'', $urls) . '\')';

		$conn   = $this->em->getConnection();
		$result = [];

		$texts = [];
		foreach ($conn->fetchAllAssociative("SELECT vct.id, vct.url, vct.h1, vct.description, vct.long_description, vct.page_title, vct.page_description
		FROM eshop_advanced_feature__virtual_category_text vct
		INNER JOIN eshop_advanced_feature__virtual_category vc ON vct.id = vc.id AND vc.site_ident = :siteIdent
		WHERE vct.locale = :locale AND vct.url {$urlQuery}", [
			'locale'    => $locale,
			'siteIdent' => $this->sites->getCurrentSite()->getIdent(),
		]) as $virtualCategory) {
			$texts[$virtualCategory['id']] = $virtualCategory;
		}

		if (!$texts) {
			return [];
		}

		$inQuery = count($texts) === 1
			? '= ' . key($texts)
			: 'IN (' . implode(',', array_keys($texts)) . ')';

		foreach ($conn->fetchAllAssociative("SELECT vc.id,
    		GROUP_CONCAT(DISTINCT cat.category_id) as cats, 
    		GROUP_CONCAT(DISTINCT manu.manufacturer_id) as manu, 
    		GROUP_CONCAT(DISTINCT f.feature_value_id) as features
		FROM eshop_advanced_feature__virtual_category vc
		LEFT JOIN eshop_advanced_feature__virtual_category_categories cat ON cat.virtual_category_id = vc.id
		LEFT JOIN eshop_advanced_feature__virtual_category_manufacturers manu ON manu.virtual_category_id = vc.id
		LEFT JOIN eshop_advanced_feature__virtual_category_features f ON f.virtual_category_id = vc.id
		WHERE vc.id {$inQuery}
		GROUP BY vc.id") as $virtualCategory) {
			$virtualCategory += $texts[$virtualCategory['id']];
			if (Strings::contains((string) $virtualCategory['cats'], ',')) {
				$virtualCategory['cats'] = explode(',', (string) $virtualCategory['cats']);
			}

			if ($virtualCategory['manu']) {
				$virtualCategory['manu'] = explode(',', (string) $virtualCategory['manu']);
			} else {
				unset($virtualCategory['manu']);
			}

			if ($virtualCategory['features']) {
				$features = [];

				foreach (explode(',', (string) $virtualCategory['features']) as $v) {
					$v = $this->features->getFeatureValueById($v);

					if ($v) {
						$features[$v['featureId']][] = (int) $v['id'];
					}
				}

				$virtualCategory['features'] = array_map(static fn($v) => implode('|', $v), $features);
			} else {
				unset($virtualCategory['features']);
			}

			if (Config::load('allowRelatedCategories', false)) {
				$virtualCategory['related'] = $this->getRelated()[$virtualCategory['id']];
			}

			$result[$virtualCategory['url']] = $virtualCategory;
		}

		return $result;
	}

	protected function getRelated(): array
	{
		if ($this->cRelated === null) {
			$this->cRelated = [];

			foreach ($this->em->createQueryBuilder()->select('IDENTITY(vcr.virtualCategory) as virtualCategoryId, vcr.targetId, vcr.targetKey')
				         ->from(VirtualCategoryRelated::class, 'vcr')
				         ->orderBy('vcr.virtualCategory', 'ASC')->addOrderBy('vcr.position', 'ASC')
				         ->getQuery()->getArrayResult() as $row) {
				$this->cRelated[$row['virtualCategoryId']][$row['targetId']] = $row;
			}
		}

		return $this->cRelated;
	}

	public function findNavigationsByGroups(array $groups): array
	{
		$result = [];

		if (!$groups) {
			return $result;
		}

		foreach ($this->getEr()->createQueryBuilder('n')->select('n.id, nt.h1, nt.url, GROUP_CONCAT(IDENTITY(g.group)) as groups, n.icon, nt.menuTitle')
			         ->innerJoin('n.texts', 'nt', Join::WITH, 'nt.locale = :lang')
			         ->innerJoin('n.inGroups', 'g', Join::WITH, 'g.group IN (:groups)')
			         ->setParameters([
				         'lang'   => $this->translator->getLocale(),
				         'groups' => $groups,
			         ])
			         ->groupBy('n.id')
			         ->orderBy('g.group')->addOrderBy('g.position', 'asc')
			         ->getQuery()->getArrayResult() as $row) {
			foreach (explode(',', $row['groups']) as $group) {
				$result[$group][$row['id']] = $row;
			}
		}

		return $result;
	}

	public function createCategoryDaoFromArray(array $data, DaoNavigationItem $nav): Category
	{
		$childs = [];

		foreach ($data as $row) {
			$dao        = new Category();
			$dao->id    = $row['id'];
			$dao->name  = $row['menuTitle'] ?: $row['h1'];
			$dao->image = $row['icon'];
			$dao->link  = '/' . ltrim($row['url'], '/');

			$childs[$row['id']] = $dao;
		}

		$category        = new Category();
		$category->id    = $nav->getId();
		$category->name  = $nav->title;
		$category->link  = $nav->link;
		$category->image = $nav->ico;

		$category->setChild($childs);

		return $category;
	}

	public function checkForNew($prepared): void
	{
		if (!empty($prepared)) {
			$siteIdent = $this->sites->getCurrentSite()->getIdent();
			$locale    = $this->translator->getLocale();

			$exists = $this->getNavigationsIdByUrlMultiple(array_keys($prepared));

			$missing = array_diff_key($prepared, $exists);

			if (empty($missing)) {
				return;
			}

			// Vyhledani bez relaci bez jazyka
			$byRelations = $this->getNavigationDataByRelations(array_diff_key($missing, $exists));

			if (!empty($byRelations)) {
				$insert = [];
				// Vlozeni jazyku
				foreach ($byRelations as $url => $id) {
					$insert[] = sprintf("(%u, '%s', '%s')", $id, $locale, $url);
				}

				$this->em->getConnection()->executeQuery("INSERT IGNORE INTO eshop_advanced_feature__virtual_category_text (id, locale, url)
					VALUES " . implode(', ', $insert));

				$missing = array_diff_key($missing, $byRelations);
			}

			// Vlozeni zakladnich dat
			$needFlush = false;
			foreach ($missing as $url => $data) {
				$needFlush = true;
				$vc        = new VirtualCategory($siteIdent);

				foreach ($data['categories'] as $id) {
					$vc->categories->add($this->em->getReference(\EshopCatalog\Model\Entities\Category::class, $id));
				}

				foreach ($data['featureValues'] as $id) {
					$vc->featureValues->add($this->em->getReference(FeatureValue::class, $id));
				}

				foreach ($data['manufacturers'] as $id) {
					$vc->manufacturers->add($this->em->getReference(Manufacturer::class, $id));
				}

				$this->em->persist($vc);

				$vct = new VirtualCategoryText($vc, $locale, $url);
				$this->em->persist($vct);
			}

			if ($needFlush) {
				$this->em->flush();
			}
		}
	}
}
