<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\AdminModule\Components\Group;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\Controls\SaveCancelControl;
use Doctrine\ORM\Query\Expr\Join;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategories;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategoryGroups;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryGroup;
use EshopAdvancedFeature\Model\GroupsCache;
use Nette\Application\UI\InvalidLinkException;

class GroupsGrid extends BaseControl
{
	public function __construct(
		protected VirtualCategoryGroups $virtualCategoryGroups,
		protected IGroupFormFactory     $formFactory,
		protected VirtualCategories     $virtualCategories,
		protected IGroupDetailFactory   $detailFactory,
		protected GroupsCache           $groupsCache,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ============= Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('eshopAdvancedFeature.title.addVirtualCategoryGroup');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this['form']->id = $id;
		$this['form']->setVirtualCategoryGroup($id);
		$this->template->modalTitle = $this->t('eshopAdvancedFeature.title.editVirtualCategoryGroup',
			['title' => $this['form']->group->getText()->title]);
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	/**
	 * @param int|string|int[]|string[] $id
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->presenter;
		$error     = false;

		if (is_array($id)) {
			$this->em->beginTransaction();
			foreach ($id as $v) {
				if (!$this->virtualCategoryGroups->remove((int) $v)) {
					$error = true;
				}
			}
			if ($error) {
				$this->em->rollback();
			} else {
				$this->em->commit();
			}
		} else {
			$this->virtualCategoryGroups->remove((int) $id);
		}

		$this->groupsCache->removeBase();

		if (!$error) {
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		$presenter->redrawControl('flashes');
		$this['grid']->reload();
	}

	/**
	 * @param int|string $id
	 *
	 * @throws InvalidLinkException
	 */
	public function handleShowDetail($id): void
	{
		$group = $this->virtualCategoryGroups->get((int) $id);

		if (!$group) {
			return;
		}

		$this['groupDetail']->sortLink = $this->link('groupDetail:sort!');
		$this['groupDetail']->groupId  = $group->getId();
		$this->template->modalTitle    = $group->getText()->title;
		$this->template->modal         = 'groupDetail';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ============= Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->virtualCategoryGroups->getEr()->createQueryBuilder('g')
			->addSelect('gt')
			->leftJoin('g.texts', 'gt', Join::WITH, 'gt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->orderBy('gt.title');

		$grid->setDataSource($qb);

		$columnPrefix = 'eshopAdvancedFeature.virtualCategoryGroup.';

		// Columns
		$grid->addColumnText('title', $columnPrefix . 'title')
			->setRenderer(fn(VirtualCategoryGroup $row) => $row->getText($this->translator->getLocale())->title);

		// Action
		$grid->addAction('detail', 'eshopAdvancedFeature.virtualCategoryGroup.sortNavigations', 'showDetail!')
			->setIcon('sort')
			->addClass('ajax');
		$grid->addAction('edit', '', 'edit!')
			->setIcon('edit')
			->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	protected function createComponentForm(): GroupForm
	{
		$control = $this->formFactory->create();

		if ($this->getParameter('id')) {
			$control->setVirtualCategoryGroup((int) $this->getParameter('id'));
		}

		$control->onAnchor[] = function() use ($control): void {
			$control['form']->onSuccessSave[] = function(): void {
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->onSuccessSaveAndClose[] = function(): void {
				$this->presenter->payload->hideModal = true;
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
			};

			/** @var SaveCancelControl $saveControl */
			$saveControl = $control['form']->getComponent('saveControl');
			$saveControl->closeModalOnCancel();
		};

		return $control;
	}

	protected function createComponentGroupDetail(): GroupDetail { return $this->detailFactory->create(); }
}
