<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\AdminModule\Components\VirtualCategory;

use Core\AdminModule\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Group;
use DynamicModule\Model\Repository\Groups;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategories;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategoryGroups;
use EshopAdvancedFeature\Model\Entities\VirtualCategory;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryInGroup;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryText;
use EshopAdvancedFeature\Model\GroupsCache;
use EshopAdvancedFeature\Model\Helpers\VirtualCategoryHelper;
use EshopAdvancedFeature\Model\VirtualCategories as DefaultVirtualCategories;
use EshopAdvancedFeature\Model\VirtualCategoriesCache;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeatureValues;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\FrontModule\Model\FilterService as FrontFilterService;
use EshopCatalog\Model\Config;
use Exception;
use Navigations\AdminModule\Model\Navigations;
use Nette\Application\Attributes\Persistent;
use Nette\Caching\Cache;
use Nette\DI\Container;
use Nette\Forms\Controls\SubmitButton;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class VirtualCategoryForm extends BaseControl
{
	#[Persistent]
	public ?int $id = null;

	public ?VirtualCategory $virtualCategory = null;

	public function __construct(
		public string                      $siteIdent,
		protected Container                $container,
		protected VirtualCategories        $virtualCategories,
		protected DefaultVirtualCategories $defaultVirtualCategories,
		protected Sites                    $sites,
		protected Features                 $features,
		protected FeatureValues            $featureValues,
		protected Manufacturers            $manufacturers,
		protected Categories               $categories,
		protected VirtualCategoryGroups    $groupService,
		protected FrontFilterService       $frontFilterService,
		protected VirtualCategoriesCache   $virtualCategoriesCache,
		protected GroupsCache              $groupsCache,
		protected Navigations              $navigations,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$titlePrefix = 'eshopAdvancedFeature.virtualCategory.';
		$form->addText('url', $titlePrefix . 'url')
			->setIsMultilanguage();
		$form->addFilesManager('icon', $titlePrefix . 'image');
		$form->addText('h1', $titlePrefix . 'h1')
			->setIsMultilanguage();
		$form->addEditor('description', $titlePrefix . 'shortDescription')
			->setDisableAutoP(false)
			->setHeight(100)
			->setIsMultilanguage();
		$form->addEditor('longDescription', $titlePrefix . 'longDescription')
			->setDisableAutoP(false)
			->setHeight(100)
			->setIsMultilanguage();
		$form->addText('filteringTitle', $titlePrefix . 'filteringTitle')
			->setIsMultilanguage();
		$form->addText('menuTitle', $titlePrefix . 'menuTitle')
			->setIsMultilanguage();

		$form->addText('pageTitle', $titlePrefix . 'pageTitle')
			->setIsMultilanguage();
		$form->addTextArea('pageDescription', $titlePrefix . 'pageDescription')
			->setIsMultilanguage();
		$form->addBool('addToSitemap', $titlePrefix . 'addToSitemap');

		$form->addCheckboxList('groups', $this->t($titlePrefix . 'groups'), $this->groupService->getOptionsForSelect())
			->setTranslator(null);
		$form->addCheckboxNestedList('categories', $this->t($titlePrefix . 'categories'), $this->categories->getFlatTree($this->siteIdent))
			->setTranslator(null);
		$form->addCheckboxList('manufacturers', $this->t($titlePrefix . 'manufacturers'), $this->manufacturers->getOptionsForSelect())
			->setTranslator(null);

		$navigations = $this->navigations->getOptionsForSelectParentsInSite($this->siteIdent);

		$form->addSortableCheckboxList('topNavigationsInCategory', $titlePrefix . 'topNavigationsInCategory', $navigations);
		$form->addSortableCheckboxList('bottomNavigationsInCategory', $titlePrefix . 'bottomNavigationsInCategory', $navigations);

		// Dalsi obsah z dynamickeho modulu
		if (class_exists(Config::class)) {
			$dynamicModuleCategoryDetail = (array) Config::load('category.dynamicModuleCategoryDetailLink');
			if ($dynamicModuleCategoryDetail && $dynamicModuleCategoryDetail['module'] && class_exists(DynamicModuleConfig::class)) {
				/** @var Groups|null $dmGroups */
				$dmGroups = $this->container->hasService('dynamicmodule.admin.groups')
					? $this->container->getService('dynamicmodule.admin.groups')
					: null;

				if ($dmGroups) {
					$qb = $dmGroups->getQueryBuilderByModule($dynamicModuleCategoryDetail['module'])
						->addOrderBy('g.lft', 'ASC');

					if (!DynamicModuleConfig::load('multiLangPublication')) {
						$qb->andWhere('g.isPublished = 1');
					}

					$list = [];

					foreach ($qb->getQuery()->getResult() as $g) {
						/** @var Group $g */
						$prefix = [];
						$p      = $g->getParent();

						while ($p) {
							if ($p->lvl <= 0) {
								break;
							}

							$prefix[] = $p->title;
							$p        = $p->getParent();
						}

						$list[$g->getId()] = ($prefix !== [] ? implode(' > ', array_reverse($prefix)) . ' > ' : '') . $g->title;
					}

					$form->addSortableCheckboxList('topDynamicModuleInCategory', 'eshopCatalog.navigation.topDynamicModuleInCategory', $list);
					$form->addSortableCheckboxList('bottomDynamicModuleInCategory', 'eshopCatalog.navigation.bottomDynamicModuleInCategory', $list);
				}
			}
		}

		// Features
		$flat         = [];
		$featureTexts = $this->features->getOptionsForSelect();
		foreach ($this->featureValues->getOptionsForSelect() as $featureId => $values) {
			$flat[] = [
				'id'     => 'f' . $featureId,
				'name'   => $featureTexts[$featureId],
				'parent' => 0,
			];

			foreach ($values as $valueId => $value) {
				$flat[] = [
					'id'     => $valueId,
					'name'   => $value,
					'parent' => 'f' . $featureId,
				];
			}
		}
		$form->addCheckboxNestedList('featureValues', $titlePrefix . 'featureValues', $flat);

		$saveCancelControl = $form->addSaveCancelControl();

		/** @var SubmitButton $saveControl */
		$saveControl = $saveCancelControl->getComponent('save');

		/** @var SubmitButton $saveAndCloseControl */
		$saveAndCloseControl = $saveCancelControl->getComponent('saveAndClose');

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$this->em->beginTransaction();
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var VirtualCategoryText[] $texts */
			$texts           = [];
			$oldCacheKeys    = [];
			$groupsCacheKeys = [];

			if ($this->id) {
				$virtualCategory = $this->virtualCategories->get($this->id);
				$texts           = $virtualCategory ? $virtualCategory->texts->toArray() : [];

				foreach ($virtualCategory->inGroups as $inGroup) {
					foreach (array_keys($langValues) as $l) {
						$groupsCacheKeys[] = GroupsCache::getKey((int) $inGroup->group->getId(), $l);
						$oldCacheKeys[]    = 'navByGroups/' . $l . '/' . $inGroup->group->getId();
					}
				}

				$relationData = [
					VirtualCategory::keyCategories    => array_map(static fn($v): int => (int) $v, $virtualCategory->categories->getKeys()),
					VirtualCategory::keyManufacturers => array_map(static fn($v): int => (int) $v, $virtualCategory->manufacturers->getKeys()),
					VirtualCategory::keyFeatureValues => array_map(static fn($v): int => (int) $v, $virtualCategory->featureValues->getKeys()),
				];

				$relationData = $this->defaultVirtualCategories->validateRelationData($relationData);
				if ($relationData === null) {
					throw new Exception('Bad relation data');
				}

				foreach (array_keys($langValues) as $l) {
					$oldCacheKeys[] = VirtualCategoryHelper::getUrlCacheKey(
						$this->siteIdent,
						$l,
						$relationData,
					);
				}
			} else {
				$virtualCategory = new VirtualCategory($this->siteIdent);
			}

			foreach ($langValues as $l => $v) {
				if ($v['url'] && !\str_starts_with((string) $v['url'], '/')) {
					$v['url'] = '/' . $v['url'];
				}

				if (!isset($texts[$l])) {
					if ($v['url']) {
						$texts[$l] = new VirtualCategoryText($virtualCategory, $l, $v['url']);
					} else {
						continue;
					}
				}

				if (!$v['url']) {
					$this->em->remove($texts[$l]);

					unset($texts[$l]);
					continue;
				}

				$texts[$l]->setUrl($v['url']);
				foreach ([
					         'h1', 'description', 'longDescription',
					         'menuTitle', 'pageTitle', 'pageDescription',
				         ] as $col) {
					$texts[$l]->{$col} = $v[$col];
				}

				$this->em->persist($texts[$l]);
				$virtualCategory->texts->set($l, $texts[$l]);
			}

			// Groups
			$formVals   = array_flip($values->groups);
			$entityVals = $virtualCategory->inGroups->toArray();
			foreach (array_diff_key($formVals, $entityVals) as $key => $value) {
				$entity = new VirtualCategoryInGroup(
					$this->groupService->getReference($key),
					$virtualCategory,
				);

				foreach (array_keys($langValues) as $l) {
					$groupsCacheKeys[] = GroupsCache::getKey((int) $entity->group->getId(), $l);
					$oldCacheKeys[]    = 'navByGroups/' . $l . '/' . $entity->group->getId();
				}

				$this->em->persist($entity);
			}

			foreach (array_diff_key($entityVals, $formVals) as $value => $inGroup) {
				$virtualCategory->inGroups->removeElement($inGroup);
				$this->em->remove($inGroup);
			}

			// Categories
			$formVals   = array_flip($values->categories);
			$entityVals = $virtualCategory->categories->toArray();
			foreach (array_diff_key($formVals, $entityVals) as $key => $value) {
				$virtualCategory->categories->set($key, $this->categories->getReference((int) $key));
			}

			foreach (array_diff_key($entityVals, $formVals) as $value) {
				$virtualCategory->categories->removeElement($value);
			}

			// Manufacturers
			$formVals   = array_flip($values->manufacturers);
			$entityVals = $virtualCategory->manufacturers->toArray();
			foreach (array_diff_key($formVals, $entityVals) as $key => $value) {
				$virtualCategory->manufacturers->set($key, $this->manufacturers->getReference((int) $key));
			}

			foreach (array_diff_key($entityVals, $formVals) as $value) {
				$virtualCategory->manufacturers->removeElement($value);
			}

			// Features
			$formVals = [];
			foreach ($values->featureValues as $value) {
				if (\str_starts_with($value, 'f')) {
					continue;
				}

				$formVals[$value] = $value;
			}
			$entityVals = $virtualCategory->featureValues->toArray();
			foreach (array_diff_key($formVals, $entityVals) as $key => $value) {
				$virtualCategory->featureValues->set($key, $this->featureValues->getReference((int) $key));
			}

			foreach (array_diff_key($entityVals, $formVals) as $value) {
				$virtualCategory->featureValues->removeElement($value);
			}

			$virtualCategory->icon         = $values->icon;
			$virtualCategory->addToSitemap = (int) $values->addToSitemap;
			$virtualCategory->setParam('eshopCatalog.topNavigationsInCategory', $values->topNavigationsInCategory);
			$virtualCategory->setParam('eshopCatalog.topDynamicModuleInCategory', $values->topDynamicModuleInCategory);
			$virtualCategory->setParam('eshopCatalog.bottomNavigationsInCategory', $values->bottomNavigationsInCategory);
			$virtualCategory->setParam('eshopCatalog.bottomDynamicModuleInCategory', $values->bottomDynamicModuleInCategory);

			$virtualCategory->lock();

			$relationData = [
				VirtualCategory::keyCategories    => array_map(static fn($v): int => (int) $v, $virtualCategory->categories->getKeys()),
				VirtualCategory::keyManufacturers => array_map(static fn($v): int => (int) $v, $virtualCategory->manufacturers->getKeys()),
				VirtualCategory::keyFeatureValues => array_map(static fn($v): int => (int) $v, $virtualCategory->featureValues->getKeys()),
			];

			$relationData = $this->defaultVirtualCategories->validateRelationData($relationData);
			if ($relationData === null) {
				throw new Exception('Bad relation data');
			}

			$virtualCategory->relationHash = $this->defaultVirtualCategories->createRelationHash($virtualCategory->siteIdent, $relationData);

			$this->em->persist($virtualCategory);
			$this->em->flush();
			$this->em->commit();

			foreach ($oldCacheKeys as $key) {
				$this->virtualCategoriesCache->getCache()->remove($key);
			}

			foreach (array_unique($groupsCacheKeys) as $key) {
				$this->groupsCache->getCache()->remove($key);
			}

			$this->virtualCategoriesCache->getCache()->clean([
				Cache::Tags => ['navByGroups'],
			]);
			$this->virtualCategoriesCache->getCache()->clean([
				Cache::Tags => ['eshopNavigation'],
			]);
			$this->virtualCategoriesCache->getCache()->clean([
				Cache::Tags => ['navigation'],
			]);

			$cache = new Cache($this->cacheStorage);
			$cache->clean([
				Cache::Tags => ['eshopNavigation', 'navigation'],
			]);

			foreach ($langValues as $l => $v) {
				if ($v['url']) {
					$this->virtualCategoriesCache->getCache()->save(VirtualCategoryHelper::getUrlCacheKey(
						$this->siteIdent,
						$l,
						$relationData,
					), $v['url'], [
						Cache::EXPIRATION => '1 week',
					]);
				}
			}

			$this->cacheStorage->clean([Cache::TAGS => ['eshopNavigation', 'navigation']]);

			$cache = new Cache($this->cacheStorage, Navigations::CACHE_NAMESPACE);
			$cache->clean([Cache::TAGS => [Navigations::CACHE_NAMESPACE]]);

			$form->addCustomData('virtualCategoryId', $virtualCategory->getId());
			$this->presenter->flashMessageSuccess('default.saved');
		} catch (Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}

			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function getVirtualCategory(): ?VirtualCategory
	{
		if (!$this->virtualCategory && $this->id) {
			$this->virtualCategory = $this->virtualCategories->get($this->id);
		}

		return $this->virtualCategory;
	}

	public function setVirtualCategory(int $id): void
	{
		$this->id = $id;
		$vc       = $this->virtualCategories->get($id);

		if (!$vc) {
			throw new InvalidArgumentException('');
		}

		$this->virtualCategory = $vc;

		$d = [
			'icon'                          => $vc->icon,
			'groups'                        => $vc->inGroups->getKeys(),
			'featureValues'                 => $vc->featureValues->getKeys(),
			'manufacturers'                 => $vc->manufacturers->getKeys(),
			'categories'                    => $vc->categories->getKeys(),
			'addToSitemap'                  => $vc->addToSitemap,
			'topNavigationsInCategory'      => $vc->getParam('eshopCatalog.topNavigationsInCategory') ?: [],
			'topDynamicModuleInCategory'    => $vc->getParam('eshopCatalog.topDynamicModuleInCategory') ?: [],
			'bottomNavigationsInCategory'   => $vc->getParam('eshopCatalog.bottomNavigationsInCategory') ?: [],
			'bottomDynamicModuleInCategory' => $vc->getParam('eshopCatalog.bottomDynamicModuleInCategory') ?: [],
		];

		foreach ($vc->texts as $lang => $text) {
			$d['h1'][$lang]              = $text->h1;
			$d['description'][$lang]     = $text->description;
			$d['longDescription'][$lang] = $text->longDescription;
			$d['menuTitle'][$lang]       = $text->menuTitle;
			$d['pageTitle'][$lang]       = $text->pageTitle;
			$d['pageDescription'][$lang] = $text->pageDescription;
			$d['url'][$lang]             = $text->getUrl();
		}

		$this['form']->setDefaults($d);
	}
}
