<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\FrontModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use Core\Model\Sites;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\Query\Parameter;
use EshopAdvancedFeature\Model\Entities\VirtualCategory;
use EshopAdvancedFeature\Model\Helpers\VirtualCategoryHelper;
use EshopAdvancedFeature\Model\Navigation\VirtualCategoryGroup;
use EshopAdvancedFeature\Model\PrefixSuffixGenerator;
use EshopAdvancedFeature\Model\VirtualCategories;
use EshopAdvancedFeature\Model\VirtualCategories as DefaultVirtualCategories;
use EshopAdvancedFeature\Model\VirtualCategoriesCache;
use EshopCatalog\FrontModule\Components\Navigation;
use EshopCatalog\FrontModule\Components\ProductsFilter;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopCatalog\FrontModule\Model\Event\CategoryHeadEvent;
use EshopCatalog\FrontModule\Model\Event\FilterLinkEvent;
use EshopCatalog\FrontModule\Presenters\DefaultPresenter;
use EshopCatalog\Model\Config;
use Navigations\Model\NavigationConfig;
use Nette\Caching\Cache;
use Nette\Http\Request;
use Nette\Http\Url;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class VirtualCategoriesSubscriber implements EventSubscriberInterface
{
	protected array $cParentPath = [];

	public function __construct(
		protected EntityManagerDecorator   $entityManager,
		protected Translator               $translator,
		protected EventDispatcher          $eventDispatcher,
		protected PrefixSuffixGenerator    $prefixSuffixGenerator,
		protected Categories               $categories,
		protected DefaultVirtualCategories $defaultVirtualCategories,
		protected Request                  $request,
		protected VirtualCategories        $virtualCategories,
		protected Sites                    $sites,
		protected VirtualCategoriesCache   $virtualCategoriesCache,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			'eshopCatalog.default.actionCategory.head'    => ['actionCategoryHead', 100],
			ProductsFilter::class . '::createFilterLink'  => 'createFilterLink',
			ProductsFilter::class . '::createSortLink'    => 'createSortLink',
			Navigation::class . '::onAttach'              => 'navigationOnAttach',
			Categories::class . '::loadRelated'           => 'loadRelated',
			ProductsFilter::class . '::beforeLoadFilters' => 'beforeLoadFilters',
		];
	}

	public function actionCategoryHead(CategoryHeadEvent $event): void
	{
		$filters = $event->activeFilters;

		$data = $this->prefixSuffixGenerator->generate(
			$filters['feature'] ?? [],
			$filters['manu'] ?? [],
			$filters['tags'] ?? [],
			$filters['priceRange'] ?? [],
			$filters['range'] ?? [],
			$event->category->id,
		);

		$activeNav    = $event->presenter->getActiveNavigation();
		$allowRelated = Config::load('allowRelatedCategories', false);
		if ($allowRelated && $activeNav && $activeNav->getParam('virtualCategoryData')) {
			$virtualCategory = $activeNav->getParam('virtualCategoryData');
			$related         = [];

			if (isset($virtualCategory['related'])) {
				$event2 = new Event([
					'related' => &$virtualCategory['related'],
					'lang'    => $activeNav->lang,
				]);

				$this->eventDispatcher->dispatch($event2, Categories::class . '::loadRelated');
				foreach ($virtualCategory['related'] as $row) {
					if ($row['targetKey'] === 'category') {
						$cat = $this->categories->get($row['targetId']);
						if ($cat instanceof Category) {
							$row['targetEntity'] = $cat;
						}
					}

					if (isset($row['targetEntity'])) {
						$related[] = $row['targetEntity'];
					}
				}
			}

			$event->category->related = $related;
		}

		$vcTexts = $activeNav->getParam('virtualCategoryData') ?? [];

		$event->categoryName = $vcTexts['h1']
			?: $this->prefixSuffixGenerator->addPrefixSuffix($event->categoryName, $data['prefix'], $data['suffix']);

		if ($vcTexts['description']) {
			$event->description                = $vcTexts['description'];
			$event->category->shortDescription = $vcTexts['description'];
		}

		if ($vcTexts['long_description']) {
			$event->category->description = $vcTexts['long_description'];
		}

		$event->title = $vcTexts['page_title']
			?: $this->prefixSuffixGenerator->addPrefixSuffix($event->title, $data['prefix'], $data['suffix']);

		if ($vcTexts['page_description']) {
			$event->metaDescription = $vcTexts['page_description'];
		}
	}

	public function createFilterLink(FilterLinkEvent $event): void
	{
		if ($event->categories === [] || count($event->categories) > 1) {
			return;
		}

		$filterFeature     = (array) ($event->currentFilter['feature'] ?? []);
		$filtersManu       = (array) ($event->currentFilter['manu'] ?? []);
		$filtersAv         = (array) ($event->currentFilter['av'] ?? []);
		$filtersTags       = (array) ($event->currentFilter['tag'] ?? []);
		$filtersPriceRange = (array) ($event->currentFilter['priceRange'] ?? []);
		$filtersRange      = (array) ($event->currentFilter['range'] ?? []);

		if ($filtersTags || $filtersPriceRange || $filtersRange) {
			return;
		}

		$createUrl  = true;
		$categoryId = $event->categories[0];
		$category   = $this->categories->get($categoryId);

		switch ($event->type) {
			case 'feature':
				if (isset($filterFeature[$event->key]) && Arrays::contains($filterFeature[$event->key], $event->value)) {
					$key = array_search($event->value, $filterFeature[$event->key], true);
					unset($filterFeature[$event->key][$key]);

					if (count($filterFeature[$event->key]) === 0) {
						unset($filterFeature[$event->key]);
					}
				} else {
					$filterFeature[$event->key][] = (int) $event->value;

					if (count($filterFeature[$event->key]) > 1) {
						$createUrl = false;
					}
				}

				break;
			case 'manu':
				if (Arrays::contains($filtersManu, $event->value)) {
					$key = array_search($event->value, $filtersManu, true);
					unset($filtersManu[$key]);
				} else {
					$filtersManu[] = (int) $event->value;

					if (count($filtersManu) > 1) {
						$createUrl = false;
					}
				}
				break;
			case 'tag':
				$createUrl = false;
				break;
			case 'av':
				if (!Arrays::contains($filtersAv, $event->value)) {
					$createUrl = false;
				}
				break;
		}

		if ($filtersAv && !Arrays::contains($filtersAv, $event->value)) {
			$createUrl = false;
		}

		if (!$category || count($filtersManu) > 1) {
			$createUrl = false;
		}

		if ($filtersManu === [] && $filterFeature === []) {
			$createUrl = false;
		}

		foreach ($filterFeature as $vals) {
			if (count($vals) > 1) {
				$createUrl = false;
			}
		}

		if ($createUrl) {
			$featureValuesIds = [];
			foreach ($filterFeature as $vals) {
				foreach ($vals as $v) {
					$featureValuesIds[] = $v;
				}
			}

			$relationData = [
				VirtualCategory::keyCategories    => $event->categories,
				VirtualCategory::keyManufacturers => $filtersManu,
				VirtualCategory::keyFeatureValues => $featureValuesIds,
			];

			$relationData = $this->virtualCategories->validateRelationData($relationData);

			if ($relationData === null) {
				$event->result = null;

				return;
			}

			$cacheCheckExist = $this->virtualCategoriesCache->getCache()->load(VirtualCategoryHelper::getUrlCacheKey(
				$this->sites->getCurrentSite()->getIdent(),
				$this->translator->getLocale(),
				$relationData,
			));

			if ($cacheCheckExist) {
				if (NavigationConfig::load('useEndSlash')) {
					$cacheCheckExist = rtrim((string) $cacheCheckExist, '/') . '/';
				}

				$event->result = $cacheCheckExist;
			} else {
				$data = $this->prefixSuffixGenerator->generate(
					$filterFeature,
					$filtersManu,
					$filtersTags,
					$filtersPriceRange,
					$filtersRange,
					$categoryId,
					$event->result,
				);

				if ($data['isMultiple']) {
					$event->result = null;

					return;
				}

				$catAlias = null;
				/** @phpstan-ignore-next-line */
				if (count($event->categories) === 1) {
					$cat = $this->categories->get($event->categories[0]);
					if ($cat instanceof Category) {
						$catAlias = $cat->alias;
					}
				}

				$baseUrl = $this->request->getUrl()->getPath();

				$lastPart = '';
				if ($categoryId) {
					$tmp      = trim($baseUrl, '/');
					$tmp      = explode('/' . $this->translator->translate('default.urlPart.page'), $tmp)[0];
					$tmp      = explode('/', $tmp);
					$tmpAlias = array_pop($tmp);
					$lastPart = $catAlias ?: $tmpAlias;
					$lastPart = trim($lastPart, '/');
				}

				$lastPart = \Nette\Utils\Strings::webalize($this->prefixSuffixGenerator->addPrefixSuffix(
					$lastPart,
					$data['prefix'] . ' ',
					' ' . $data['suffix'])
				);
				$lastPart = trim($lastPart, '-');

				$tmp[] = $lastPart;

				$event->result = '/' . ltrim(implode('/', $tmp), '/');

				if ($event->result !== rtrim($this->request->getUrl()->getPath(), '/')) {
					VirtualCategoryHelper::prepareCreateVirtualCategory(
						$event->result,
						[$categoryId],
						$filtersManu,
						array_unique(array_merge(...$filterFeature)),
						$this->sites->getCurrentSite()->getIdent(),
						$this->translator->getLocale(),
					);
					$this->virtualCategoriesCache->getCache()->save(VirtualCategoryHelper::getUrlCacheKey(
						$this->sites->getCurrentSite()->getIdent(),
						$this->translator->getLocale(),
						$relationData,
					), $event->result, [
						Cache::Expire => '30 minutes',
					]);
				}
			}

			if (isset($event->currentFilter['sort'])) {
				$tmpUrl = new Url($event->result);
				$tmpUrl->setQueryParameter('productsFilter-sort', urlencode($event->currentFilter['sort']));
				$event->result = $tmpUrl->getAbsoluteUrl();
			}
		} else {
			$event->result = null;
		}
	}

	public function createSortLink(FilterLinkEvent $event): void
	{
		if (empty($event->currentFilter) || array_key_exists('priceRange', $event->currentFilter)) {
			return;
		}

		$url = new Url($this->request->getUrl());
		if ($event->value && $event->value !== 'recommended') {
			$url->setQueryParameter('productsFilter-sort', urlencode($event->value));
		} else {
			$url->setQueryParameter('productsFilter-sort', null);
		}

		$rangeParams = [];
		foreach ($event->currentFilter['range'] ?? [] as $k => $vals) {
			$rangeParams[$k] = implode('|', $vals);
		}

		if ($rangeParams !== []) {
			$url->setQueryParameter('productsFilter-fr', $rangeParams);
		} else {
			$url->setQueryParameter('productsFilter-fr', null);
		}

		$event->result = $url->getAbsoluteUrl();
	}

	protected function getParentPath(Category $category): string
	{
		$key = $category->getId() . '_' . $this->translator->getLocale();

		if (!isset($this->cParentPath[$key])) {
			$this->cParentPath[$key] = implode('/', array_map(
					static fn(Category $c) => $c->getAttr('originAlias') ?: $c->alias,
					array_reverse($category->getParentPath())
				)
			);
		}

		return $this->cParentPath[$key];
	}

	public function navigationOnAttach(ControlEvent $event): void
	{
		/** @var Navigation $control */
		$control     = $event->control;
		$groupsInNav = [];

		foreach ($control->getNavs() as $nav) {
			if ($nav->componentType !== VirtualCategoryGroup::COMPONENT_TYPE) {
				continue;
			}

			$nav->componentParams['category']            = 'virtualCategory_' . $nav->componentParams['group'];
			$groupsInNav[$nav->componentParams['group']] = $nav;
		}

		foreach ($this->defaultVirtualCategories->findNavigationsByGroups(array_keys($groupsInNav)) as $groupId => $navs) {
			if ($navs) {
				$control->otherCategories['virtualCategory_' . $groupId] = $this->defaultVirtualCategories->createCategoryDaoFromArray($navs, $groupsInNav[$groupId]);
			}
		}
	}

	public function loadRelated(Event $event): void
	{
		$ids = [];

		foreach ($event->data['related'] as $k => $rows) {
			if (isset($rows['targetKey'])) {
				if ($rows['targetKey'] === 'virtualCategory') {
					$ids[$rows['targetId']] = [
						'k1' => $k,
					];
				}
			} else {
				foreach ($rows as $k2 => $row) {
					if ($row['targetKey'] !== 'virtualCategory') {
						continue;
					}

					$ids[$row['targetId']] = [
						'k1' => $k,
						'k2' => $k2,
					];
				}
			}
		}

		if ($ids !== []) {
			foreach ($this->defaultVirtualCategories->getEr()->createQueryBuilder('vc')
				         ->select('vc.id, vct.url, vct.h1, vc.icon, vct.description, vct.longDescription')
				         ->innerJoin('vc.texts', 'vct', Join::WITH, 'vct.locale = :locale')
				         ->where('vc.id IN (:ids)')
				         ->setParameters(new ArrayCollection([new Parameter('locale', $event->data['lang']), new Parameter('ids', array_keys($ids))]))->getQuery()->getArrayResult() as $row) {
				if (!$row['url']) {
					continue;
				}

				$cat                   = new Category;
				$cat->name             = $row['h1'];
				$cat->nameH1           = $row['h1'];
				$cat->alias            = $cat->name ? Strings::webalize($cat->name) : '';
				$cat->link             = $row['url'];
				$cat->image            = $row['icon'];
				$cat->defaultImage     = $row['icon'];
				$cat->shortDescription = $row['description'];
				$cat->description      = $row['longDescription'];

				$k = $ids[$row['id']];
				if (isset($k['k2'])) {
					$event->data['related'][$k['k1']][$k['k2']]['targetEntity'] = $cat;
				} else {
					$event->data['related'][$k['k1']]['targetEntity'] = $cat;
				}
			}
		}
	}

	public function beforeLoadFilters(ControlEvent $event): void
	{
		/** @var ProductsFilter $control */
		$control = $event->control;
		/** @var DefaultPresenter|null $presenter */
		$presenter = $control->getPresenterIfExists();

		if (!$presenter) {
			return;
		}

		$nav          = $presenter->getActiveNavigation();
		$groupFilters = $nav->getParam('virtualCategoryData')['groups'][$nav->componentParams['group']]['features'] ?? null;

		if ($groupFilters) {
			$presenter['productsFilter']->allowedFeatures = $groupFilters;
		}
	}
}
