<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Entities\TranslateListener;
use Core\Model\TemplateReader\Entity\ITemplate;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\Manufacturer;
use Nette\Utils\DateTime;

#[ORM\Table('eshop_advanced_feature__virtual_category')]
#[ORM\Index(name: 'eafvc_site', columns: ['site_ident'])]
#[ORM\Entity]
#[ORM\EntityListeners([TranslateListener::class])]
class VirtualCategory implements ITemplate
{
	use TId;

	public const keyCategories    = 'categories';
	public const keyManufacturers = 'manufacturers';
	public const keyFeatureValues = 'featureValues';

	/** @var Collection<string, VirtualCategoryText> */
	#[ORM\OneToMany(targetEntity: VirtualCategoryText::class, mappedBy: 'virtualCategory', indexBy: 'locale')]
	public Collection $texts;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['unsigned' => true, 'default' => 0])]
	protected int $isLocked = 0;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['unsigned' => true, 'default' => 0])]
	public int $addToSitemap = 0;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: true)]
	public ?string $icon = null;

	#[ORM\Column(type: Types::STRING, nullable: false)]
	public string $siteIdent;

	/** @var Collection<Category> */
	#[ORM\JoinTable(name: 'eshop_advanced_feature__virtual_category_categories')]
	#[ORM\JoinColumn(name: 'virtual_category_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'category_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToMany(targetEntity: Category::class, indexBy: 'id')]
	public Collection $categories;

	/** @var Collection<Manufacturer> */
	#[ORM\JoinTable(name: 'eshop_advanced_feature__virtual_category_manufacturers')]
	#[ORM\JoinColumn(name: 'virtual_category_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'manufacturer_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToMany(targetEntity: Manufacturer::class, indexBy: 'id')]
	public Collection $manufacturers;

	/** @var Collection<FeatureValue> */
	#[ORM\JoinTable(name: 'eshop_advanced_feature__virtual_category_features')]
	#[ORM\JoinColumn(name: 'virtual_category_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'feature_value_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToMany(targetEntity: FeatureValue::class, indexBy: 'id')]
	public Collection $featureValues;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, options: ['default' => 'CURRENT_TIMESTAMP'])]
	public \DateTime $created;

	/** @var Collection<VirtualCategoryRelated> */
	#[ORM\OneToMany(targetEntity: VirtualCategoryRelated::class, mappedBy: 'virtualCategory')]
	public Collection $related;

	/** @var Collection<VirtualCategoryInGroup> */
	#[ORM\OneToMany(targetEntity: VirtualCategoryInGroup::class, mappedBy: 'virtualCategory', indexBy: 'group_id')]
	public Collection $inGroups;

	#[ORM\Column(type: Types::STRING, length: 50, nullable: true)]
	public ?string $relationHash = null;

	#[ORM\Column(name: 'last_visit', type: Types::DATETIME_MUTABLE, nullable: false, options: ['default' => 'CURRENT_TIMESTAMP'])]
	public ?\DateTime $lastVisit = null;

	#[ORM\Column(name: 'params', type: Types::JSON, nullable: true)]
	private ?array $params = null;

	public function __construct(
		string $siteIdent
	)
	{
		$this->siteIdent     = $siteIdent;
		$this->created       = new DateTime;
		$this->texts         = new ArrayCollection;
		$this->categories    = new ArrayCollection;
		$this->manufacturers = new ArrayCollection;
		$this->featureValues = new ArrayCollection;
		$this->related       = new ArrayCollection;
		$this->inGroups      = new ArrayCollection;
		$this->lastVisit     = new \DateTime();
	}

	public function lock(bool $lock = true): void { $this->isLocked = $lock ? 1 : 0; }

	public function isLocked(): bool { return (bool) $this->isLocked; }

	/**
	 * @param string $locale
	 */
	public function getText($locale = null): ?VirtualCategoryText
	{
		if ($locale) {
			return $this->texts[$locale] ?? null;
		}

		return $this->texts->first() ?: null;
	}

	public function setTexts($texts): self
	{
		$this->texts = new ArrayCollection((array) $texts);

		return $this;
	}

	public function getTexts() { return $this->texts->toArray(); }

	public function getParams(): array { return $this->params ?: []; }

	/**
	 * @return mixed|null $key
	 */
	public function getParam(string $key) { return $this->params[$key] ?? null; }

	/**
	 * @param mixed|null $value
	 */
	public function setParam(string $key, $value): self
	{
		if ($value) {
			if (!is_array($this->params)) {
				$this->params = [];
			}

			$this->params[$key] = $value;
		} else {
			unset($this->params[$key]);
		}

		return $this;
	}
}
