<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model;

use Contributte\Translation\Exceptions\InvalidArgument;
use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Manufacturer;
use EshopCatalog\FrontModule\Model\Manufacturers;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use Nette\Caching\Cache;
use Nette\Utils\Strings;

class PrefixSuffixGenerator
{
	protected ?array $cFeatureData        = null;
	protected ?array $cCategoryRody       = null;
	protected array  $cFeatureValuesTexts = [];

	public function __construct(
		protected EntityManagerDecorator $entityManager,
		protected Manufacturers          $manufactures,
		protected Translator             $translator,
		protected CacheService           $cacheService,
	)
	{
	}

	/**
	 * @param int $categoryId
	 *
	 * @throws InvalidArgument
	 */
	public function generate(
		array   $features,
		array   $manufactures,
		array   $tags,
		array   $priceRange,
		array   $ranges = [],
		        $categoryId = null,
		?string $text = null
	): array
	{
		$rod = $categoryId ? $this->getCategoryRod($categoryId) : null;

		$result = [
			'prefix'       => '',
			'suffix'       => '',
			'text'         => $text,
			'isMultiple'   => false,
			'priceRange'   => false,
			'range'        => false,
			'usedFeatures' => [],
			'allParts'     => [],
		];

		if ($features !== []) {
			$featureIds = array_keys($features);
			$valueIds   = array_merge(...$features);

			$featuresR = [
				'prefix' => [],
				'suffix' => [],
			];
			$values    = $this->getFeatureValueTexts($this->translator->getLocale(), $valueIds);

			$featuresAdvanced     = $this->getFeatureData($featureIds);
			$result['isMultiple'] = count($featureIds) !== count($featuresAdvanced);

			foreach ($featuresAdvanced as $row) {
				if (!isset($features[$row['id']])) {
					$result['isMultiple'] = true;
				}

				if (count($features[$row['id']]) > 1) {
					$result['isMultiple'] = true;
					$tmp                  = [
						'position' => $row['beaUrlPositionMulti'],
						'priority' => (int) $row['beaUrlPriorityMulti'],
						'text'     => $row['beaUrlMulti'],
					];
				} else {
					$featureRod = $rod ? 'rod' . $rod : 'name';

					$tmp = array_values($features[$row['id']]);

					$featureText = $values[$tmp[0]][$featureRod] ?: null;

					if (!$featureText) {
						$featureText = $values[$tmp[0]]['name'];
					}

					$tmp = [
						'position' => $row['beaUrlPosition'],
						'priority' => (int) $row['beaUrlPriority'],
						'text'     => trim(str_replace('%v%', (string) $featureText, $row['beaUrlPattern'])),
					];
				}
				$result['usedFeatures'][]      = (int) $row['id'];
				$tmp['feature']                = (int) $row['id'];
				$featuresR[$tmp['position']][] = $tmp;
			}

			usort($featuresR['prefix'], static fn(array $a, array $b): int => $a['priority'] <=> $b['priority']);

			foreach ($featuresR as $position => $group) {
				foreach ($group as $feature) {
					$result['allParts'][] = $feature['text'];
					$result[$position]    .= ' ' . $feature['text'];
				}
			}
		}

		if ($manufactures !== []) {
			if (count($manufactures) > 1) {
				$result['prefix']     .= ' ' . $this->translator->translate('eshopAdvancedFeatureFront.moreManufacturers.' . strtolower($rod ?: 'z'));
				$result['isMultiple'] = true;
			} else {
				$manu = $this->manufactures->get((int) array_values($manufactures)[0]);
				if ($manu instanceof Manufacturer) {
					$result['suffix'] .= ' ' . $manu->name;
				}
			}
		}

		if ($tags !== []) {
			$result['isMultiple'] = true;
		}

		if ($ranges !== []) {
			$result['range'] = true;
		}

		if (isset($priceRange['min']) || isset($priceRange['max'])) {
			$result['priceRange'] = true;
		}

		if (isset($priceRange['min'])) {
			$result['suffix'] .= ' ' . $this->translator->translate('eshopAdvancedFeatureFront.priceFrom', [
					'price' => number_format((float) $priceRange['min'], 0, '', ' '),
				]) . ' kč';
		}

		if (isset($priceRange['max'])) {
			$result['suffix'] .= ' ' . $this->translator->translate('eshopAdvancedFeatureFront.priceTo', [
					'price' => number_format((float) $priceRange['max'], 0, '', ' '),
				]) . ' kč';
		}

		if ($text) {
			$result['text'] = $this->addPrefixSuffix($text, $result['prefix'], $result['suffix']);
		}

		return $result;
	}

	public function normalizeText(string $text): string
	{
		$expl = explode(' ', $text);
		$tmp  = substr($expl[0], 1);

		if ($tmp && $tmp === strtolower($tmp)) {
			$text = str_replace($expl[0], strtolower($expl[0]), $text);
		}

		return $text;
	}

	public function addPrefixSuffix(string $text, string $prefix = '', string $suffix = ''): string
	{
		if ($prefix !== '' && $prefix !== '0') {
			$text = Strings::firstLower($text);
		}

		return ucfirst(trim($prefix . ' ' . $this->normalizeText($text) . $suffix));
	}

	protected function getFeatureValueTexts(string $lang, array $valueIds): array
	{
		if (!array_key_exists($lang, $this->cFeatureValuesTexts)) {
			$this->cFeatureValuesTexts[$lang] = $this->cacheService->defaultCache->load('featureValueTexts_' . $lang, function(&$dep) use ($lang) {
				$dep = [
					Cache::TAGS => ['features'],
				];
				$arr = [];

				$qb = $this->entityManager->getRepository(FeatureValueTexts::class)->createQueryBuilder('fvt')
					->select('IDENTITY(fvt.id) as id, fvt.name, fvt.rodM, fvt.rodZ, fvt.rodS')
					->andWhere('fvt.lang = :lang')
					->setParameters(new ArrayCollection([new Parameter('lang', $lang)]))->getQuery();

				foreach ($qb->getScalarResult() as $row) {
					$arr[$row['id']] = $row;
				}

				return $arr;
			});
		}

		$result = [];
		foreach ($valueIds as $valId) {
			if (isset($this->cFeatureValuesTexts[$lang][$valId])) {
				$result[$valId] = $this->cFeatureValuesTexts[$lang][$valId];
			}
		}

		return $result;
	}

	protected function getFeatureData(array $featureIds): array
	{
		if ($this->cFeatureData === null) {
			$lang = $this->translator->getLocale();

			$this->cFeatureData = $this->cacheService->defaultCache->load('featureData/' . $lang, function(&$dep) {
				$dep = [
					Cache::Expire => '1 week',
				];

				$arr = [];
				foreach ($this->entityManager->getRepository(Feature::class)->createQueryBuilder('f')
					         ->select('f.id, f.beaUrlPosition, f.beaUrlPriority, f.beaUrlPattern, f.beaUrlPositionMulti, f.beaUrlPriorityMulti')
					         ->addSelect('ft.beaUrlMulti')
					         ->andWhere('f.beautyUrlActive = 1')
					         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')->setParameter('lang', $this->translator->getLocale())
					         ->orderBy('f.beaUrlPriority', 'ASC')->getQuery()->getArrayResult() as $row) {
					$arr[$row['id']] = $row;
				}

				return $arr;
			});
		}

		$result = [];
		foreach ($this->cFeatureData as $k => $v) {
			if (in_array($k, $featureIds)) {
				$result[$k] = $v;
			}
		}

		return $result;
	}

	protected function getCategoryRod(int $id): string
	{
		if ($this->cCategoryRody === null) {
			$this->cCategoryRody = $this->cacheService->categoryCache->load('categoryRod', function(&$dep) {
				$dep = [
					Cache::Tags       => [Categories::CACHE_NAMESPACE, 'categories'],
					Cache::EXPIRATION => '1 week',
				];

				$arr = [];
				foreach ($this->entityManager->getRepository(Category::class)->createQueryBuilder('c')
					         ->select('c.id, c.rod')
					         ->andWhere('c.rod IS NOT NULL')
					         ->getQuery()->getArrayResult() as $row) {
					$arr[$row['id']] = strtoupper((string) $row['rod']);
				}

				return $arr;
			});
		}

		return $this->cCategoryRody[$id] ?? 'z';
	}
}
