<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Components\Navigation\DaoNavigationItem;
use Core\FrontModule\Model\Redirects;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Caster;
use Core\Model\Sites;
use EshopAdvancedFeature\Model\Entities\VirtualCategory;
use EshopAdvancedFeature\Model\Helpers\VirtualCategoryHelper;
use EshopAdvancedFeature\Model\Navigation\VirtualCategoryGroup;
use EshopAdvancedFeature\Model\VirtualCategories;
use EshopCatalog\Model\Navigation\BaseNavigation;
use EshopCatalog\Model\Navigation\Home;
use Navigations\Model\Event\RouteInEvent;
use Navigations\Model\Event\RouteOutEvent;
use Navigations\Model\Navigations;
use Nette\Http\Request;
use Nette\Utils\DateTime;
use Nette\Utils\Json;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class VirtualCategoriesNavigationSubscriber implements EventSubscriberInterface
{
	public static array $lastVisitUpdated = [];

	public function __construct(
		protected EntityManagerDecorator $em,
		protected VirtualCategories      $virtualCategories,
		protected Navigations            $navigations,
		protected Sites                  $sites,
		protected Translator             $translator,
		protected Request                $request,
		protected Redirects              $redirects,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			Navigations::class . '::routeInNotFoundPreRedirect' => ['routeInNotFoundPreRedirect', 110],
			BaseNavigation::class . '::routeOut'                => ['routeOut', 110],
		];
	}

	public function routeInNotFoundPreRedirect(RouteInEvent $event): void
	{
		$url    = '/' . ltrim((string) $event->urlParams['path'], '/');
		$locale = $event->urlParams['locale'];

		$data = $this->virtualCategories->getNavigationDataByUrl($url, $locale);

		if ($data !== []) {
			$featureValuesIds = [];
			foreach ($data['features'] as $feature) {
				$featureValuesIds[] = (int) $feature;
			}

			$dataCheck = [
				VirtualCategory::keyCategories    => array_map(static fn($v): int => (int) $v, explode(',', (string) $data['cats'])),
				VirtualCategory::keyManufacturers => array_map(static fn($v): int => (int) $v, $data['manu'] ?: []),
				VirtualCategory::keyFeatureValues => $featureValuesIds,
			];

			$dataCheck     = $this->virtualCategories->validateRelationData($dataCheck);
			$dataCheckHash = $this->virtualCategories->createRelationHash($this->sites->getCurrentSite()->getIdent(), $dataCheck);

			if ($data['relation_hash'] !== $dataCheckHash && (int) $data['is_locked'] !== 1) {
				$correctData = $this->virtualCategories->getUrlByHash($this->translator->getLocale(), $dataCheckHash);

				if ($correctData === []) {
					$this->virtualCategories->updateHash($data['id'], $dataCheckHash);
				} else {
					$this->redirects->addNew($url, $correctData['url'], $this->sites->getCurrentSite()->getIdent(), $locale);
					$this->virtualCategories->removeRow($data['id']);

					$urlEntity = $this->request->getUrl();
					$urlEntity->withPath($url);

					header('HTTP/1.1 301 Moved Permanently');
					header('Location: ' . $urlEntity->getAbsoluteUrl());
					exit;
				}
			}

			$filter = [];
			$return = [
				'presenter' => 'EshopCatalog:Front:Default',
				'action'    => 'category',
			];

			if ($data['cats']) {
				$return += [
					'id' => is_array($data['cats']) ? implode('|', $data['cats']) : (int) $data['cats'],
				];
			}

			if ($data['manu']) {
				$filter['fm'][0] = implode('|', $data['manu']);
			}

			if ($data['features']) {
				$filter['ff'] = $data['features'];
			}

			if ($filter !== []) {
				$return += [
					'productsFilter-filter' => $filter,
				];
			}

			/** @var DaoNavigationItem|null $activeNav */
			$activeNav  = null;
			$activeNavs = $this->navigations->getPublishedByComponent(VirtualCategoryGroup::COMPONENT_TYPE, $locale);
			foreach ($activeNavs as $nav) {
				foreach ($data['groups'] ?? [] as $g) {
					if ($g['id'] === (int) $nav->componentParams['group']) {
						$activeNav = $nav;
						break 2;
					}
				}
			}

			if (!$activeNav) {
				$tmp       = $this->navigations->getPublishedByComponent(Home::COMPONENT_NAME, $locale);
				$activeNav = array_shift($tmp);
			}

			if ($activeNav) {
				if (is_scalar($data['params'])) {
					try {
						$data['params'] = Json::decode(Caster::toString($data['params']), forceArrays: true);

						foreach ($data['params'] as $k => $v) {
							$activeNav->setParam($k, $v);
						}
					} catch (\Exception $e) {
					}
				}

				$activeNav->setParam('virtualCategory', $data['id']);
				$activeNav->setParam('virtualCategoryData', $data);
				$activeNav->setParam('url', $url);

				$return['activeNavigation'] = $activeNav;
			}

			if (
				!array_key_exists($data['id'], self::$lastVisitUpdated)
				&& $data['last_visit']
				&& $data['last_visit'] < (new DateTime())->modify('-1 day')->format('Y-m-d H:i:s')
			) {
				self::$lastVisitUpdated[$data['id']] = $data['id'];
				VirtualCategoryHelper::updateLastVisit($data['id']);
			}

			$event->return = $return;
			$event->stopPropagation();
		}
	}

	public function routeOut(RouteOutEvent $event): void
	{
		$activeNav = $event->navigation;

		if ($activeNav->getParam('virtualCategory')) {
			unset($event->urlParams['productsFilter-filter']);

			if ($event->urlParams['productsFilter-fr'] === null) {
				unset($event->urlParams['productsFilter-fr']);
			}
		}
	}
}
