<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\AdminModule\Components\Group;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategories;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategoryGroups;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryInGroup;
use EshopAdvancedFeature\Model\GroupsCache;
use EshopAdvancedFeature\Model\VirtualCategoriesCache;
use Navigations\AdminModule\Model\Navigations;
use Nette\Caching\Cache;
use Nette\Http\Request;

class GroupDetail extends BaseControl
{
	/** @var int|null @persistent */
	public ?int $groupId = null;

	public ?string $sortLink = null;

	protected VirtualCategories      $virtualCategories;
	protected VirtualCategoryGroups  $virtualCategoryGroups;
	protected Request                $request;
	protected VirtualCategoriesCache $virtualCategoriesCache;
	protected GroupsCache            $groupsCache;

	public function __construct(
		VirtualCategories      $virtualCategories,
		VirtualCategoryGroups  $virtualCategoryGroups,
		Request                $request,
		VirtualCategoriesCache $virtualCategoriesCache,
		GroupsCache            $groupsCache
	)
	{
		$this->virtualCategories      = $virtualCategories;
		$this->virtualCategoryGroups  = $virtualCategoryGroups;
		$this->request                = $request;
		$this->virtualCategoriesCache = $virtualCategoriesCache;
		$this->groupsCache            = $groupsCache;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setItemsPerPageList([], true);

		$qb = $this->em->getRepository(VirtualCategoryInGroup::class)->createQueryBuilder('vcg')
			->addSelect('vc, vct')
			->where('vcg.group = :group')
			->innerJoin('vcg.virtualCategory', 'vc')
			->innerJoin('vc.texts', 'vct')
			->setParameter('group', $this->groupId)
			->orderBy('vcg.position', 'asc');

		$grid->setDataSource($qb)
			->setSortable()
			->setSortableHandler($this->sortLink);

		$grid->addColumnText('title', 'default.title')
			->setRenderer(static function(VirtualCategoryInGroup $group): string {
				return $group->virtualCategory->getText()->h1;
			});
		$grid->addColumnText('url', 'eshopAdvancedFeature.virtualCategory.url')
			->setRenderer(static function(VirtualCategoryInGroup $group): string {
				return $group->virtualCategory->getText()->getUrl();
			});

		return $grid;
	}

	public function handleSort(): void
	{
		$presenter = $this->presenter;
		$groupId   = $this->request->getPost('id', null);
		$position  = $this->request->getPost('position', null);
		/** @var VirtualCategoryInGroup $group */
		$group = $this->em->getRepository(VirtualCategoryInGroup::class)->find($groupId);

		if ($groupId != null && $position != null && $group !== null) {
			$group->position = (int) $position;
			$this->em->persist($group);
			$this->em->flush();
			$presenter->flashMessageSuccess('default.positionChanged');

			foreach ($this->langsService->getLangs(false) as $v) {
				$this->groupsCache->getCache()->remove(GroupsCache::getKey((int) $group->getId(), $v->getTag()));
				$this->virtualCategoriesCache->getCache()->remove('navByGroups/' . $v->getTag() . '/' . $group->getId());
			}

			$this->virtualCategoriesCache->getCache()->clean([
				Cache::Tags => ['navByGroups'],
			]);
			$this->virtualCategoriesCache->getCache()->clean([
				Cache::Tags => ['eshopNavigation'],
			]);
			$this->virtualCategoriesCache->getCache()->clean([
				Cache::Tags => ['navigation'],
			]);

			$cache = new Cache($this->cacheStorage);
			$cache->clean([
				Cache::Tags => ['eshopNavigation', 'navigation'],
			]);

			$this->cacheStorage->clean([Cache::TAGS => ['eshopNavigation', 'navigation']]);

			$cache = new Cache($this->cacheStorage, Navigations::CACHE_NAMESPACE);
			$cache->clean([Cache::TAGS => [Navigations::CACHE_NAMESPACE]]);
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$presenter->redrawControl('flashes');
	}

}
