<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\AdminModule\Components\VirtualCategory;

use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategories;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategoryGroups;
use EshopAdvancedFeature\Model\Entities\VirtualCategory;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryText;
use EshopAdvancedFeature\Model\GroupsCache;
use EshopAdvancedFeature\Model\VirtualCategoriesCache;
use Exception;
use Navigations\AdminModule\Model\Navigations;
use Nette\Application\UI\InvalidLinkException;
use Nette\Caching\Cache;
use Nette\ComponentModel\IComponent;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;

class VirtualCategoryTextForm extends TemplateReaderControl
{
	/** @var int|null @persistent */
	public ?int $id = null;

	/** @var VirtualCategory|null */
	protected                        $entity;
	protected VirtualCategories      $virtualCategoriesService;
	protected VirtualCategoryGroups  $virtualCategoryGroupsService;
	protected GroupsCache            $groupsCache;
	protected VirtualCategoriesCache $virtualCategoriesCache;

	public function __construct(
		VirtualCategories      $virtualCategories,
		VirtualCategoryGroups  $virtualCategoryGroups,
		GroupsCache            $groupsCache,
		VirtualCategoriesCache $virtualCategoriesCache
	)
	{
		$this->virtualCategoriesService     = $virtualCategories;
		$this->virtualCategoryGroupsService = $virtualCategoryGroups;
		$this->groupsCache                  = $groupsCache;
		$this->virtualCategoriesCache       = $virtualCategoriesCache;
	}

	public function render(): void
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];
		$this->template->virtualCategory    = $this->entity;

		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @param IComponent $presenter
	 *
	 * @throws InvalidLinkException
	 */
	protected function attached($presenter): void
	{
		parent::attached($presenter);
		$httpRequest = $this->presenter->getHttpRequest();

		$this->setFields();
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addComponent(new BaseContainer(), 'component');

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var VirtualCategoryText[] $texts */
			$texts           = [];
			$groupsCacheKeys = [];
			$componentValues = $form->getComponent('component')->getValues(true);

			if ($this->getVirtualCategory()) {
				$entity = $this->entity;
				$texts  = $entity->texts->toArray();
			} else {
				throw new Exception('Virtual category not found');
			}


			foreach ($entity->inGroups as $inGroup) {
				foreach ($this->langsService->getLangs(false) as $l => $v) {
					$groupsCacheKeys[] = GroupsCache::getKey((int) $inGroup->group->getId(), $l);
					$groupsCacheKeys[] = 'navByGroups/' . $l . '/' . $inGroup->group->getId();
				}
			}

			if (!$langValues) {
				foreach ($this->langsService->getLangs(false) as $l => $v) {
					$langValues[$l]['component'] = $componentValues;
				}
			}

			foreach ($langValues as $l => $v) {
				$v = $v['component'];
				if (!isset($texts[$l])) {
					continue;
				}

				foreach ($v as $c => $cValue) {
					/** @phpstan-ignore-next-line */
					$comp = $form['component'][$c];
					if ($comp instanceof BaseContainer === false) {
						continue;
					}

					$tmp = Json::decode(Json::encode($comp->getValues()), Json::FORCE_ARRAY);

					$v[$c] = array_merge($tmp, $cValue);

					if (array_key_exists('types', $tmp) && array_key_exists('types', $cValue)) {
						$v[$c]['types'] = array_merge($tmp['types'], $cValue['types']);
					}
				}

				foreach (array_diff(array_keys($componentValues), array_keys($v)) as $name) {
					if (is_array($componentValues[$name]) && array_key_exists($l, $componentValues[$name])) {
						$v[$name] = $componentValues[$name][$l];
					} else {
						$v[$name] = $componentValues[$name];
					}
				}

				$texts[$l]->setTexts($v);
				$this->em->persist($texts[$l]);
			}

			$flashMessage = $entity->getId() ? 'pages.templateForm.edited' : 'pages.templateForm.added';

			$this->em->flush();

			foreach ($entity->inGroups as $group) {
				foreach ($this->langsService->getLangs(false) as $lang) {
					$this->groupsCache->removeGroup($group->getId(), $lang->getTag());
				}
			}

			foreach (array_unique($groupsCacheKeys) as $key) {
				$this->virtualCategoriesCache->getCache()->remove($key);
				$this->groupsCache->getCache()->remove($key);
			}

			$this->virtualCategoriesCache->getCache()->clean([
				Cache::Tags => ['navByGroups'],
			]);
			$this->virtualCategoriesCache->getCache()->clean([
				Cache::Tags => ['eshopNavigation'],
			]);
			$this->virtualCategoriesCache->getCache()->clean([
				Cache::Tags => ['navigation'],
			]);

			$cache = new Cache($this->cacheStorage);
			$cache->clean([
				Cache::Tags => ['eshopNavigation', 'navigation'],
			]);

			$cache = new Cache($this->cacheStorage, Navigations::CACHE_NAMESPACE);
			$cache->clean([Cache::TAGS => [Navigations::CACHE_NAMESPACE]]);

			$this->groupsCache->removeBase();

			$form->addCustomData('virtualCategoryId', $entity->getId());
			$this->presenter->flashMessageSuccess($flashMessage);
		} catch (Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function getVirtualCategory(): ?VirtualCategory
	{
		if (!$this->entity && $this->id) {
			$this->entity = $this->virtualCategoriesService->get($this->id);
		}

		return $this->entity;
	}

	public function setVirtualCategory(int $id): void
	{
		$this->entity = $this->virtualCategoriesService->get($id);

		if (!$this->entity) {
			$this->presenter->error();
		}

		$this->id = $id;

		$form = $this['form'];
		$d    = [];
	}

	protected function setFields(): void
	{
		$parsedTemplate = null;
		foreach ($this->getVirtualCategory()->inGroups as $v) {
			if ($v->group->templatePageItem) {
				$parsedTemplate = $this->parseTemplate($v->group->templatePageItem);
			}
		}

		if ($parsedTemplate['template'] && $this->httpRequest->getQuery('do') != 'virtualCategoryTextForm-loadInputs') {
			$tmp = explode('|', $parsedTemplate['template']);
			if (count($tmp) === 2) {
				$this->templateReader->setTemplatesDir($this->pathsService->getTemplatePagesDir($tmp[0]));
			}

			/** @var BaseContainer $container */
			$container = $this['form']['component'];
			$this->templateReader->loadTemplateComponents($container, $parsedTemplate['file'], $this->getVirtualCategory()->texts->toArray());
			$this->templateReader->setDefaults($container, $this->getVirtualCategory()->texts->toArray(), $parsedTemplate['file']);
		}
	}

	protected function parseTemplate(string $template): array
	{
		$result = [
			'template' => $template,
			'file'     => $template,
		];

		if (strpos($template, '|') === false) {
			$key                = 'default|' . $template;
			$result['template'] = isset($this['form']->getComponent('template')->getItems()[$key]) ? $key : 'default' . '|' . $template;
		} else {
			$result['file'] = explode('|', $template)[1] ?? $template;
		}

		return $result;
	}
}
