<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\AdminModule\Model\Subscribers;

use Core\Model\UI\Form\BootstrapRenderer;
use EshopCatalog\Model\Config;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\Entities\EntityManagerDecorator;
use EshopCatalog\AdminModule\Components\Features\FeatureForm;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\Feature;

class FeatureFormSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $entityManager;
	protected CacheService           $cacheService;

	protected static ?string $featureType = null;

	public function __construct(
		EntityManagerDecorator $entityManager,
		CacheService           $cacheService
	)
	{
		$this->entityManager = $entityManager;
		$this->cacheService  = $cacheService;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			FeatureForm::class . '::createForm'  => ['createForm', 100],
			FeatureForm::class . '::formSuccess' => ['formSuccess', 100],
			FeatureForm::class . '::setFeature'  => ['setFeature', 100],
		];
	}

	public function createForm(CreateFormEvent $event): void
	{
		$form = $event->form;

		$form->addGroup('eshopAdvancedFeature.featureForm.title');
		$form->addBool('beautyUrlActive', 'eshopAdvancedFeature.feature.isActive')
			->setDefaultValue(0);
		$form->addBool('beaUrlPosition', 'eshopAdvancedFeature.feature.oneIsPrefix', [
			0 => 'eshopAdvancedFeature.feature.beforeTitle',
			1 => 'eshopAdvancedFeature.feature.afterTitle',
		]);
		$form->addText('beaUrlPriority', 'eshopAdvancedFeature.feature.prefixPriority')
			->setDefaultValue(10)->setHtmlType('number');
		$form->addText('beaUrlMulti', 'eshopAdvancedFeature.feature.multi')->setIsMultilanguage();
		$form->addBool('beaUrlPositionMulti', 'eshopAdvancedFeature.feature.multiIsPrefix', [
			0 => 'eshopAdvancedFeature.feature.beforeTitle',
			1 => 'eshopAdvancedFeature.feature.afterTitle',
		]);
		$form->addText('beaUrlPriorityMulti', 'eshopAdvancedFeature.feature.prefixPriorityMultiple')
			->setDefaultValue(10)->setHtmlType('number');
		$form->addHidden('beaUrlPattern', 'eshopAdvancedFeature.feature.pattern')->setDefaultValue('%v%');

		/** @var BootstrapRenderer $renderer */
		$renderer                                                           = $event->form->getRenderer();
		$renderer->extendedLayout['eshopAdvancedFeature.featureForm.title'] = __DIR__ . '/FeatureFormLayout.latte';
	}

	public function formSuccess(FormSuccessEvent $event): void
	{
		/** @var Feature $feature */
		$feature = $event->custom['entity'];
		$values  = $event->values;

		$position      = $values->beaUrlPosition ? 'suffix' : 'prefix';
		$multiPosition = $values->beaUrlPositionMulti ? 'suffix' : 'prefix';

		$feature->beautyUrlActive = $values->beautyUrlActive;
		$feature->beaUrlPosition  = $position;
		$feature->beaUrlPriority  = (int) $values->beaUrlPriority;
		$feature->beaUrlPattern   = $values->beaUrlPattern;

		$feature->beaUrlPositionMulti = $multiPosition;
		$feature->beaUrlPriorityMulti = (int) $values->beaUrlPriorityMulti;
		$this->entityManager->persist($feature);

		foreach ($feature->getFeatureTexts() as $l => $text) {
			$text->beaUrlMulti = $values->beaUrlMulti[$l] ?? null;
			$this->entityManager->persist($text);
		}

		$this->entityManager->flush();

		if (self::$featureType !== null && Config::load('features.allowType') && self::$featureType !== $values->type) {
			$featureValuesIds = [];
			foreach ($feature->getFeatureValues() as $v) {
				$featureValuesIds[] = $v->getId();
			}

			foreach (array_chunk($featureValuesIds, 200) as $ids) {
				$vcForDelete = [];
				foreach ($this->entityManager->getConnection()->fetchAllAssociative("SELECT virtual_category_id as id FROM eshop_advanced_feature__virtual_category_features WHERE feature_value_id IN (" . implode(',', $ids) . ") GROUP BY virtual_category_id") as $row) {
					$vcForDelete[] = $row['id'];
				}

				if ($vcForDelete) {
					$this->entityManager->getConnection()->executeQuery("DELETE FROM eshop_advanced_feature__virtual_category WHERE id IN (" . implode(',', $vcForDelete) . ")");
				}
			}
		}
	}

	public function setFeature(SetFormDataEvent $event): void
	{
		/** @var Feature|null $feature */
		$feature = $event->entity;

		if ($feature) {
			if (Config::load('features.allowType')) {
				self::$featureType = $feature->type;
			}

			$multi = [];
			foreach ($feature->getFeatureTexts() as $l => $text) {
				$multi[$l] = $text->beaUrlMulti;
			}

			$event->form->setDefaults([
				'beautyUrlActive'     => $feature->beautyUrlActive ? 1 : 0,
				'beaUrlPosition'      => $feature->beaUrlPosition === 'prefix' ? 0 : 1,
				'beaUrlPriority'      => $feature->beaUrlPriority,
				'beaUrlPattern'       => $feature->beaUrlPattern,
				'beaUrlMulti'         => $multi,
				'beaUrlPositionMulti' => $feature->beaUrlPositionMulti === 'prefix' ? 0 : 1,
				'beaUrlPriorityMulti' => $feature->beaUrlPriorityMulti,
			]);
		}
	}
}
