<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\AdminModule\Model;

use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use EshopAdvancedFeature\Model\Entities\VirtualCategory;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryRelated;
use EshopAdvancedFeature\Model\Helpers\VirtualCategoryHelper;
use EshopAdvancedFeature\Model\VirtualCategoriesCache;
use Exception;

/**
 * @method VirtualCategory getReference($id)
 * @method VirtualCategory[] getAll()
 * @method VirtualCategory|null get($id)
 */
class VirtualCategories extends BaseEntityService
{
	protected $entityClass = VirtualCategory::class;

	protected VirtualCategoriesCache $virtualCategoriesCache;

	public function __construct(
		VirtualCategoriesCache $virtualCategoriesCache
	)
	{
		$this->virtualCategoriesCache = $virtualCategoriesCache;
	}

	public function addRelated(int $virtualCategory, int $target, string $key = 'category'): ?VirtualCategoryRelated
	{
		try {
			/** @var VirtualCategory $vc */
			$vc                = $this->em->getReference(VirtualCategory::class, $virtualCategory);
			$entity            = new VirtualCategoryRelated($vc, $target);
			$entity->targetKey = $key;

			$this->em->persist($entity);
			$this->em->flush();

			$this->virtualCategoriesCache->clearRelated();

			return $entity;
		} catch (Exception $e) {

		}

		return null;
	}

	public function removeRelated(int $id): bool
	{
		if ($entity = $this->em->getRepository(VirtualCategoryRelated::class)->find($id)) {
			$this->em->remove($entity);
			$this->em->flush();

			$this->virtualCategoriesCache->clearRelated();

			return true;
		}

		return false;
	}

	public function getByFeatureNavigation(int $id): array
	{
		$arr = [];

		foreach ($this->em->getRepository(VirtualCategoryRelated::class)->createQueryBuilder('vcr')
			         ->where('vcr.virtualCategory = :id')
			         ->setParameter('id', $id)
			         ->orderBy('vcr.position')
			         ->getQuery()->getArrayResult() as $row) {
			$arr[$row['id']] = $row;
		}

		return $arr;
	}

	/**
	 * @param int|string $id
	 * @param int|string $position
	 */
	public function setPosition($id, $position): bool
	{
		/** @var ?VirtualCategoryRelated $item */
		$item = $this->em->getRepository(VirtualCategoryRelated::class)->find($id);

		if ($item) {
			$item->setPosition((int) $position);
			$this->em->persist($item);
			$this->em->flush();

			$this->virtualCategoriesCache->clearRelated();

			return true;
		}

		return false;
	}

	public function addToSitemap(array $ids): bool
	{
		try {
			$this->em->createQueryBuilder()
				->update(VirtualCategory::class, 'vc')
				->set('vc.addToSitemap', 1)
				->where('vc.id IN (:ids)')
				->setParameter('ids', $ids)
				->getQuery()
				->execute();

			return true;
		} catch (Exception $e) {

		}

		return false;
	}

	public function removeFromSitemap(array $ids): bool
	{
		try {
			$this->em->createQueryBuilder()
				->update(VirtualCategory::class, 'vc')
				->set('vc.addToSitemap', 0)
				->where('vc.id IN (:ids)')
				->setParameter('ids', $ids)
				->getQuery()
				->execute();

			return true;
		} catch (Exception $e) {

		}

		return false;
	}

	public function removeMultiple(array $ids): bool
	{
		$entities = $this->getEr()->findBy(['id' => $ids]);

		if (count($entities) !== count($ids)) {
			return false;
		}

		$cacheKeys = [];
		$this->em->beginTransaction();
		try {
			/** @phpstan-ignore-next-line */
			$result = Arrays::every($entities, function(VirtualCategory $entity, $k, $ids) use (&$cacheKeys): bool {
				try {
					$relationData = [
						VirtualCategory::keyCategories    => array_map(static fn($v) => (int) $v, $entity->categories->getKeys()),
						VirtualCategory::keyManufacturers => array_map(static fn($v) => (int) $v, $entity->manufacturers->getKeys()),
						VirtualCategory::keyFeatureValues => array_map(static fn($v) => (int) $v, $entity->featureValues->getKeys()),
					];

					foreach ($entity->texts->getKeys() as $l) {
						$cacheKeys[] = VirtualCategoryHelper::getUrlCacheKey(
							$entity->siteIdent,
							$l,
							$relationData,
						);
					}

					$this->em->remove($entity);

					return true;
				} catch (Exception $e) {
				}

				return false;
			});

			if ($result) {
				$this->em->flush();
				$this->em->commit();

				foreach ($cacheKeys as $key) {
					$this->virtualCategoriesCache->getCache()->remove($key);
				}
			} else {
				throw new Exception('result false');
			}
		} catch (Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}

			$result = false;
		}

		$this->virtualCategoriesCache->clearRelated();

		return $result;
	}

	/**
	 * @param int|string $id
	 *
	 * @return bool
	 */
	public function remove($id)
	{
		$cacheKeys = [];
		if ($entity = $this->get($id)) {
			$relationData = [
				VirtualCategory::keyCategories    => array_map(static fn($v) => (int) $v, $entity->categories->getKeys()),
				VirtualCategory::keyManufacturers => array_map(static fn($v) => (int) $v, $entity->manufacturers->getKeys()),
				VirtualCategory::keyFeatureValues => array_map(static fn($v) => (int) $v, $entity->featureValues->getKeys()),
			];

			foreach ($entity->texts->getKeys() as $l) {
				$cacheKeys[] = VirtualCategoryHelper::getUrlCacheKey(
					$entity->siteIdent,
					$l,
					$relationData,
				);
			}

			$this->em->remove($entity);
			$this->em->flush();

			foreach ($cacheKeys as $key) {
				$this->virtualCategoriesCache->getCache()->remove($key);
			}

			$this->virtualCategoriesCache->clearRelated();

			return true;
		}

		return false;
	}

}
