<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\CronModule\Presenters;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Arrays;
use Core\Model\Sites;
use Doctrine\ORM\Query\Expr\Join;
use EshopAdvancedFeature\Model\Entities\VirtualCategory;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryText;
use EshopAdvancedFeature\Model\Migrator;
use EshopAdvancedFeature\Model\VirtualCategories;
use EshopCatalog\FrontModule\Model\Categories;
use Nette\Application\ForbiddenRequestException;

class DefaultPresenter extends BasePresenter
{
	protected EntityManagerDecorator $em;
	protected Migrator               $migrator;
	protected Categories             $categories;
	protected Sites                  $sites;
	protected VirtualCategories      $virtualCategories;

	public function __construct(
		EntityManagerDecorator $em,
		Migrator               $migrator,
		Categories             $categories,
		Sites                  $sites,
		VirtualCategories      $virtualCategories
	)
	{
		parent::__construct();
		$this->em                = $em;
		$this->migrator          = $migrator;
		$this->categories        = $categories;
		$this->sites             = $sites;
		$this->virtualCategories = $virtualCategories;
	}

	public function actionGenerateRelationHash(): void
	{
		set_time_limit(999);
		ini_set('memory_limit', '2g');

		$categories = [];
		foreach ($this->em->getConnection()->fetchAllAssociative('SELECT * FROM eshop_advanced_feature__virtual_category_categories') as $row) {
			$categories[(int) $row['virtual_category_id']][] = (int) $row['category_id'];
		}

		$manufacturers = [];
		foreach ($this->em->getConnection()->fetchAllAssociative('SELECT * FROM eshop_advanced_feature__virtual_category_manufacturers') as $row) {
			$manufacturers[(int) $row['virtual_category_id']][] = (int) $row['manufacturer_id'];
		}

		$featureValues = [];
		foreach ($this->em->getConnection()->fetchAllAssociative('SELECT * FROM eshop_advanced_feature__virtual_category_features') as $row) {
			$featureValues[(int) $row['virtual_category_id']][] = (int) $row['feature_value_id'];
		}


		foreach ($this->em->getRepository(VirtualCategory::class)->createQueryBuilder('vc')
			         ->select('vc.id, vc.siteIdent')
			         ->where('vc.relationHash IS NULL')
			         ->getQuery()->getArrayResult() as $row) {
			if (!$row['siteIdent']) {
				continue;
			}

			$data = [
				VirtualCategory::keyCategories    => $categories[$row['id']] ?? [],
				VirtualCategory::keyManufacturers => $manufacturers[$row['id']] ?? [],
				VirtualCategory::keyFeatureValues => $featureValues[$row['id']] ?? [],
			];

			$data = $this->virtualCategories->validateRelationData($data);
			if ($data === null) {
				continue;
			}

			$this->em->getConnection()->update('eshop_advanced_feature__virtual_category', [
				'relation_hash' => $this->virtualCategories->createRelationHash($row['siteIdent'], $data),
			], [
				'id' => $row['id'],
			]);
		}

		$this->sendJson(['ok']);
	}

	public function actionNavigations(): void
	{
		ini_set('memory_limit', '4G');
		if (!$this->getUser()->isAllowed('Core:Admin', 'access')) {
			throw new ForbiddenRequestException;
		}

		$lang = $this->getParameter('lang', 'cs');
		$data = [];

		foreach ($this->em->getConnection()->fetchAllAssociative("SELECT vc.id, vc.site_ident, vct.url, vct.h1 
					FROM eshop_advanced_feature__virtual_category vc
					INNER JOIN eshop_advanced_feature__virtual_category_text vct ON vc.id = vct.id AND vct.locale = '{$lang}'
					WHERE vc.is_locked = 1") as $row) {
			$data[] = [
				'id'   => (string) $row['id'],
				'name' => $row['site_ident'] . ' -> ' . $row['h1'] . ' ' . $row['url'],
			];
		}

		$this->sendJson($data);
	}

	public function actionCheckCategories(): void
	{
		set_time_limit(1000);
		echo "<table>";
		foreach ($this->sites->getSites() as $site) {
			foreach ($site->getDomains() as $domain) {
				$catLinks = $this->categories->getAllCategoriesLinks($site->getIdent(), $domain->getLang());

				foreach ($this->em->getRepository(VirtualCategoryText::class)->createQueryBuilder('vct')
					         ->select('IDENTITY(vct.virtualCategory) as id, vct.url')
					         ->innerJoin('vct.virtualCategory', 'vc', Join::WITH, 'vc.siteIdent = :siteIdent AND vc.isLocked = 0')
					         ->where('vct.locale = :lang')
					         ->setParameters([
						         'lang'      => $domain->getLang(),
						         'siteIdent' => $site->getIdent(),
					         ])->getQuery()->getArrayResult() as $row) {
					if (Arrays::contains($catLinks, $row['url'])) {
						$badLinks[$row['id']] = $row['url'];

						echo "<tr>";
						echo "<td>{$site->getIdent()}</td>";
						echo "<td>{$domain->getLang()}</td>";
						echo "<td>{$row['id']}</td>";
						echo "<td>{$row['url']}</td>";
						echo "</tr>";
					}
				}
			}
		}

		echo "</table>";

		if ($this->getParameter('remove') === '1') {
			echo 'remove';

			if (!empty($badLinks)) {
				foreach (array_chunk(array_keys($badLinks), 200) as $chunk) {
					$this->em->getConnection()->executeQuery("DELETE FROM eshop_advanced_feature__virtual_category WHERE id IN (" . implode(',', $chunk) . ")");
				}
			}

			$this->virtualCategories->prepareUrlsByRelation();
		}

		die();
	}
}
