<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model\Entities;

use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;

/**
 * @ORM\Table("eshop_advanced_feature__feature_navigation", indexes={@ORM\Index(name="eaffn_site", columns={"site_ident"})})
 * @ORM\Entity
 */
class FeatureNavigation
{
	use TId;

	/**
	 * @var Collection<string, FeatureNavigationText>
	 * @ORM\OneToMany(targetEntity="FeatureNavigationText", mappedBy="featureNavigation", indexBy="locale")
	 */
	protected Collection $texts;

	/**
	 * @ORM\Column(type="integer", nullable=true)
	 */
	public ?int $navigationId = null;

	/**
	 * @var bool
	 * @ORM\Column(type="boolean", nullable=false, options={"default": 0})
	 */
	public $dynamicParams;

	/**
	 * @var array|ArrayHash
	 * @ORM\Column(type="array", nullable=false)
	 */
	public $params;

	/**
	 * @ORM\Column(type="smallint", nullable=false, options={"unsigned": true, "default": 0})
	 */
	protected int $isLocked = 0;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $ico = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $siteIdent = null;

	/**
	 * @var Collection<FeatureNavigationInGroup>
	 * @ORM\OneToMany(targetEntity="FeatureNavigationInGroup", mappedBy="featureNavigation", cascade={"all"}, orphanRemoval=true)
	 */
	protected Collection $groups;

	/**
	 * @var Collection<FeatureNavigationRelated>
	 * @ORM\OneToMany(targetEntity="FeatureNavigationRelated", mappedBy="featureNavigation")
	 */
	protected Collection $related;

	/**
	 * @ORM\Column(type="datetime", options={"default": "CURRENT_TIMESTAMP"})
	 */
	public ?DateTimeInterface $created = null;

	public function __construct(?int $navigationId = null, array $params)
	{
		$this->navigationId  = $navigationId;
		$this->params        = $params;
		$this->dynamicParams = false;
		$this->texts         = new ArrayCollection;
		$this->groups        = new ArrayCollection;
		$this->related       = new ArrayCollection;
		$this->created       = new DateTime('now');
	}

	public function lock(bool $lock = true): self
	{
		$this->isLocked = $lock ? 1 : 0;

		return $this;
	}

	public function isLocked(): bool { return (bool) $this->isLocked; }

	/**
	 * @return Collection<string, FeatureNavigationText>
	 */
	public function getTexts(): Collection { return $this->texts; }

	public function getText(?string $locale = null): ?FeatureNavigationText
	{
		if ($locale) {
			return $this->texts[$locale] ?? null;
		}

		return $this->texts->first() ?: null;
	}

	/**
	 * @param FeatureNavigationText[] $texts
	 */
	public function setTexts(array $texts): void
	{
		$this->texts = new ArrayCollection($texts);
	}

	/**
	 * @return Collection<FeatureNavigationInGroup>
	 */
	public function getGroups(): Collection { return $this->groups; }

	/**
	 * @param FeatureNavigationInGroup[] $groups
	 */
	public function setGroups(array $groups): void
	{
		$this->groups = new ArrayCollection($groups);
	}
}
