<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\TemplateReader\Entity\ITemplate;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\Manufacturer;
use Nette\Utils\DateTime;

/**
 * @ORM\Table("eshop_advanced_feature__virtual_category", indexes={@ORM\Index(name="eafvc_site", columns={"site_ident"})})
 * @ORM\Entity
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener"})
 */
class VirtualCategory implements ITemplate
{
	use TId;

	public const keyCategories    = 'categories';
	public const keyManufacturers = 'manufacturers';
	public const keyFeatureValues = 'featureValues';

	/**
	 * @var Collection<string, VirtualCategoryText>
	 * @ORM\OneToMany(targetEntity="VirtualCategoryText", mappedBy="virtualCategory", indexBy="locale")
	 */
	public Collection $texts;

	/**
	 * @ORM\Column(type="smallint", nullable=false, options={"unsigned": true, "default": 0})
	 */
	protected int $isLocked = 0;

	/**
	 * @ORM\Column(type="smallint", nullable=false, options={"unsigned": true, "default": 0})
	 */
	public int $addToSitemap = 0;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $icon = null;

	/**
	 * @ORM\Column(type="string", nullable=false)
	 */
	public string $siteIdent;

	/**
	 * @var Collection<Category>
	 * @ORM\ManyToMany(targetEntity="EshopCatalog\Model\Entities\Category", indexBy="id")
	 * @ORM\JoinTable(name="eshop_advanced_feature__virtual_category_categories",
	 *     joinColumns={@ORM\JoinColumn(name="virtual_category_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="category_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	public Collection $categories;

	/**
	 * @var Collection<Manufacturer>
	 * @ORM\ManyToMany(targetEntity="EshopCatalog\Model\Entities\Manufacturer", indexBy="id")
	 * @ORM\JoinTable(name="eshop_advanced_feature__virtual_category_manufacturers",
	 *     joinColumns={@ORM\JoinColumn(name="virtual_category_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="manufacturer_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	public Collection $manufacturers;

	/**
	 * @var Collection<FeatureValue>
	 * @ORM\ManyToMany(targetEntity="EshopCatalog\Model\Entities\FeatureValue", indexBy="id")
	 * @ORM\JoinTable(name="eshop_advanced_feature__virtual_category_features",
	 *     joinColumns={@ORM\JoinColumn(name="virtual_category_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="feature_value_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	public Collection $featureValues;

	/**
	 * @ORM\Column(type="datetime", options={"default": "CURRENT_TIMESTAMP"})
	 */
	public DateTimeInterface $created;

	/**
	 * @var Collection<VirtualCategoryRelated>
	 * @ORM\OneToMany(targetEntity="VirtualCategoryRelated", mappedBy="virtualCategory")
	 */
	public Collection $related;

	/**
	 * @var Collection<VirtualCategoryInGroup>
	 * @ORM\OneToMany(targetEntity="VirtualCategoryInGroup", mappedBy="virtualCategory", indexBy="group_id")
	 */
	public Collection $inGroups;

	/**
	 * @ORM\Column(type="string", length=50, nullable=true)
	 */
	public ?string $relationHash = null;

	/**
	 * @ORM\Column(name="last_visit", type="datetime", nullable=false, options={"default": "CURRENT_TIMESTAMP"})
	 */
	public ?\DateTimeInterface $lastVisit = null;

	/**
	 * @ORM\Column(name="params", type="json", nullable=true)
	 */
	private ?array $params = null;

	public function __construct(
		string $siteIdent
	)
	{
		$this->siteIdent     = $siteIdent;
		$this->created       = new DateTime;
		$this->texts         = new ArrayCollection;
		$this->categories    = new ArrayCollection;
		$this->manufacturers = new ArrayCollection;
		$this->featureValues = new ArrayCollection;
		$this->related       = new ArrayCollection;
		$this->inGroups      = new ArrayCollection;
		$this->lastVisit     = new \DateTime();
	}

	public function lock(bool $lock = true): void { $this->isLocked = $lock ? 1 : 0; }

	public function isLocked(): bool { return (bool) $this->isLocked; }

	/**
	 * @param string $locale
	 *
	 * @return VirtualCategoryText|null
	 */
	public function getText($locale = null): ?VirtualCategoryText
	{
		if ($locale) {
			return $this->texts[$locale] ?? null;
		}

		return $this->texts->first() ?: null;
	}

	public function setTexts($texts): self
	{
		$this->texts = new ArrayCollection((array) $texts);

		return $this;
	}

	public function getTexts() { return $this->texts->toArray(); }

	public function getParams(): array { return $this->params ?: []; }

	/**
	 * @return mixed|null $key
	 */
	public function getParam(string $key) { return $this->params[$key] ?? null; }

	/**
	 * @param mixed|null $value
	 */
	public function setParam(string $key, $value): self
	{
		if ($value) {
			if (!is_array($this->params)) {
				$this->params = [];
			}

			$this->params[$key] = $value;
		} else {
			unset($this->params[$key]);
		}

		return $this;
	}
}
