<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use Core\Model\Lang\Langs;
use Doctrine\ORM\Query\Expr\Join;
use EshopAdvancedFeature\Model\Entities\FeatureNavigation;
use EshopAdvancedFeature\Model\Entities\FeatureNavigationGroup;
use EshopAdvancedFeature\Model\Entities\FeatureNavigationGroupFilter;
use EshopAdvancedFeature\Model\Entities\FeatureNavigationGroupText;
use EshopAdvancedFeature\Model\Entities\FeatureNavigationInGroup;
use EshopAdvancedFeature\Model\Entities\FeatureNavigationRelated;
use EshopAdvancedFeature\Model\Entities\FeatureNavigationText;

class Migrator
{
	protected EntityManagerDecorator $em;
	protected VirtualCategories      $virtualCategories;
	protected Langs                  $langs;

	public function __construct(
		EntityManagerDecorator $em,
		VirtualCategories      $virtualCategories,
		Langs                  $langs
	)
	{
		$this->em                = $em;
		$this->virtualCategories = $virtualCategories;
		$this->langs             = $langs;
	}

	public function migrateNavigation(): array
	{
		$conn = $this->em->getConnection();
		$this->em->getConfiguration()->setSQLLogger(null);
		$oldBaseBySite = [];
		foreach ($this->em->getRepository(FeatureNavigation::class)->createQueryBuilder('fn')
			         ->where('fn.isLocked = 1')
			         ->andWhere('fn.dynamicParams = 0')
			         ->getQuery()->getArrayResult() as $row) {
			$row['siteIdent'] = (string) $row['siteIdent'];
			$features         = [];
			foreach ($row['params']['productsFilter-filter']['ff'] ?? [] as $v) {
				$features[] = explode('|', $v);
			}

			$manufactures = [];
			foreach ($row['params']['productsFilter-filter']['fm'] ?? [] as $v) {
				$manufactures[] = explode('|', $v);
			}

			$oldBaseBySite[$row['siteIdent']][$row['id']] = [
				'created'       => $row['created'],
				'siteIdent'     => $row['siteIdent'],
				'icon'          => $row['ico'],
				'categories'    => is_array($row['params']['id']) ? $row['params']['id'] : [$row['params']['id']],
				'features'      => array_merge(...$features),
				'manufacturers' => array_merge(...$manufactures),
			];
		}

		$oldNewIdsMap = [];
		foreach ($oldBaseBySite as $siteIdent => $oldBase) {
			$oldData    = [];
			$allTextIds = [];
			foreach (array_chunk(array_keys($oldBase), 200) as $ids) {
				foreach ($this->em->getRepository(FeatureNavigationText::class)->createQueryBuilder('fnt')
					         ->addSelect('IDENTITY(fnt.featureNavigation) as navigationId')
					         ->where('fnt.featureNavigation IN (' . implode(',', $ids) . ')')
					         ->andWhere('fnt.url IS NOT NULL AND fnt.url != \'\'')
					         ->getQuery()->getArrayResult() as $row) {
					$row = $row[0] + [
							'navigationId' => $row['navigationId'],
						];

					$allTextIds[$row['navigationId']]              = $row['navigationId'];
					$oldData[$row['locale']][$row['navigationId']] = [
						'url'             => '/' . ltrim($row['url'], '/'),
						'h1'              => $row['pageTitle'],
						'filteringTitle'  => $row['filteringTitle'],
						'description'     => $row['pageDescription'],
						'pageTitle'       => $row['h1'],
						'menuTitle'       => $row['menuTitle'],
						'pageDescription' => $row['description'],
						'longDescription' => $row['longDescription'],
					];
				}
			}

			ksort($oldData);

			foreach (array_diff_key($oldBase, $allTextIds) as $k => $v) {
				unset($oldBase[$k]);
			}

			foreach ($oldBase as $oldId => $row) {
				$conn->executeQuery("INSERT INTO eshop_advanced_feature__virtual_category (created, site_ident, icon, is_locked) 
				VALUES (:created, :siteIdent, :icon, :isLocked)", [
					'created'   => $row['created']->format('Y-m-d H:i:s'),
					'siteIdent' => $siteIdent,
					'icon'      => $row['icon'],
					'isLocked'  => 1,
				]);

				$virtualCategoryId = $conn->lastInsertId();

				foreach ($row['categories'] as $v) {
					$conn->executeQuery("INSERT IGNORE INTO eshop_advanced_feature__virtual_category_categories (virtual_category_id, category_id) VALUES (:vcId, :cId)", [
						'vcId' => $virtualCategoryId,
						'cId'  => $v,
					]);
				}

				foreach ($row['manufacturers'] as $v) {
					$conn->executeQuery("INSERT IGNORE INTO eshop_advanced_feature__virtual_category_manufacturers (virtual_category_id, manufacturer_id) VALUES (:vcId, :mId)", [
						'vcId' => $virtualCategoryId,
						'mId'  => $v,
					]);
				}

				foreach ($row['features'] as $v) {
					$conn->executeQuery("INSERT IGNORE INTO eshop_advanced_feature__virtual_category_features (virtual_category_id, feature_value_id) VALUES (:vcId, :fId)", [
						'vcId' => $virtualCategoryId,
						'fId'  => $v,
					]);
				}

				$oldNewIdsMap[$oldId] = $virtualCategoryId;
			}

			$loopCount = 0;
			foreach ($oldData as $locale => $navigations) {
				foreach ($navigations as $oldNavId => $row) {

					$vcId = $oldNewIdsMap[$oldNavId] ?? null;
					/** @phpstan-ignore-next-line */
					if (!$vcId || !$row['url'] || Strings::contains($row['url'], '/strana/')) {
						continue;
					}

					if ($loopCount > 0) {
						$oldBaseData = $oldBase[$oldNavId];
						$checkExist  = $this->virtualCategories->getNavigationDataByRelations(
							$siteIdent,
							[
								$row['url'] => [
									'categories'    => $oldBaseData['categories'],
									'manufacturers' => $oldBaseData['manufacturers'],
									'featureValues' => $oldBaseData['features'],
								],
							]);

						if (isset($checkExist[$row['url']])) {
							$vcId = $checkExist[$row['url']];
						}
					}

					$conn->executeQuery("INSERT IGNORE INTO eshop_advanced_feature__virtual_category_text 
    				(id, locale, url, h1, filtering_title, description, page_title, menu_title, page_description, long_description)
    				VALUES (:id, :locale, :url, :h1, :filteringTitle, :description, :pageTitle, :menuTitle, :pageDescription, :longDescription)", [
						'id'              => $vcId,
						'locale'          => $locale,
						'url'             => $row['url'],
						'h1'              => $row['h1'] ?: null,
						'filteringTitle'  => $row['filteringTitle'] ?: null,
						'description'     => $row['description'] ?: null,
						'pageTitle'       => $row['pageTitle'] ?: null,
						'menuTitle'       => $row['menuTitle'] ?: null,
						'pageDescription' => $row['pageDescription'] ?: null,
						'longDescription' => $row['longDescription'] ?: null,
					]);
				}
				$loopCount++;
			}
			$oldBase = null;
			$oldData = null;

			$oldRelated = [];
			foreach ($this->em->getRepository(FeatureNavigationRelated::class)->createQueryBuilder('r')
				         ->select('IDENTITY(r.featureNavigation) as navigation, r.target, r.targetKey, r.position')
				         ->orderBy('r.position', 'ASC')
				         ->getQuery()->getArrayResult() as $row) {
				$newId = $oldNewIdsMap[$row['navigation']];
				if (!$newId) {
					continue;
				}

				if ($row['targetKey'] === 'advancedFeature') {
					$row['targetKey'] = 'virtualCategory';
					$row['target']    = $oldNewIdsMap[$row['target']];
				}

				$oldRelated[$newId][] = [
					'targetId'  => $row['target'],
					'targetKey' => $row['targetKey'],
				];
			}

			foreach ($oldRelated as $vcId => $navigations) {
				$pos = 0;
				foreach ($navigations as $row) {
					$conn->executeQuery("INSERT INTO eshop_advanced_feature__virtual_category_related (virtual_category_id, target_id, target_key, `position`)
					VALUES (:vcId, :targetId, :targetKey, :position)", [
						'vcId'      => $vcId,
						'targetId'  => $row['targetId'],
						'targetKey' => $row['targetKey'],
						'position'  => $pos++,
					]);
				}
			}
			$oldRelated = null;
		}

		return $oldNewIdsMap;
	}

	public function migrateGroups(array $oldNavToNewVirtualCategoryMap = []): void
	{
		$oldGroups = [];

		foreach ($this->em->getRepository(FeatureNavigationGroup::class)->createQueryBuilder('g')
			         ->select('g.id, g.ico')
			         ->getQuery()->getArrayResult() as $row) {
			$oldGroups[$row['id']] = [
				'icon'        => $row['ico'],
				'texts'       => [],
				'navigations' => [],
				'filters'     => [],
			];
		}

		foreach ($this->em->getRepository(FeatureNavigationGroupText::class)->createQueryBuilder('gt')
			         ->select('IDENTITY(gt.featureNavigationGroup) as group, gt.title, gt.lang')
			         ->getQuery()->getArrayResult() as $row) {
			if (!isset($oldGroups[$row['group']])) {
				continue;
			}

			$oldGroups[$row['group']]['texts'][$row['lang']] = [
				'title' => $row['title'],
			];
		}

		foreach ($this->em->getRepository(FeatureNavigationGroupFilter::class)->createQueryBuilder('gf')
			         ->select('IDENTITY(gf.group) as group, IDENTITY(gf.feature) as feature, gf.position')
			         ->orderBy('gf.position', 'ASC')
			         ->getQuery()->getArrayResult() as $row) {
			if (!isset($oldGroups[$row['group']])) {
				continue;
			}

			$oldGroups[$row['group']]['filters'][] = [
				'feature'  => $row['feature'],
				'position' => $row['position'],
			];
		}

		foreach ($this->em->getRepository(FeatureNavigationInGroup::class)->createQueryBuilder('gn')
			         ->select('IDENTITY(gn.group) as group, IDENTITY(gn.featureNavigation) as navigation, gn.position')
			         ->orderBy('gn.position', 'ASC')
			         ->getQuery()->getArrayResult() as $row) {
			if (!isset($oldGroups[$row['group']])) {
				continue;
			}

			$oldGroups[$row['group']]['navigations'][] = [
				'navigation' => $row['navigation'],
				'position'   => $row['position'],
			];
		}

		$conn = $this->em->getConnection();
		foreach ($oldGroups as $group) {
			$conn->executeQuery("INSERT INTO eshop_advanced_feature__virtual_category_group (icon) VALUES (:icon)", [
				'icon' => $group['icon'],
			]);

			$groupId = $conn->lastInsertId();

			foreach ($group['texts'] as $lang => $text) {
				$conn->executeQuery("INSERT INTO eshop_advanced_feature__virtual_category_group_text (group_id, lang, title) VALUES (:groupId, :lang, :title)", [
					'groupId' => $groupId,
					'lang'    => $lang,
					'title'   => $text['title'],
				]);
			}

			$pos = 0;
			foreach ($group['filters'] as $filter) {
				$conn->executeQuery("INSERT INTO eshop_advanced_feature__virtual_category_group_filter (group_id, feature_id, `position`) VALUES (:groupId, :featureId, :position)", [
					'groupId'   => $groupId,
					'featureId' => $filter['feature'],
					'position'  => $pos++,
				]);
			}

			$pos = 0;
			foreach ($group['navigations'] as $navigation) {
				$conn->executeQuery("INSERT INTO eshop_advanced_feature__virtual_category_in_group (group_id, virtual_category_id, `position`) 
					VALUES (:groupId, :vcId, :position)", [
					'groupId'  => $groupId,
					'vcId'     => $oldNavToNewVirtualCategoryMap[$navigation['navigation']],
					'position' => $pos++,
				]);
			}
		}
	}

	public function migrateMissingLangsInGroups(): void
	{
		$conn = $this->em->getConnection();

		$inGroup = [];
		foreach ($conn->fetchAllAssociative("SELECT virtual_category_id as id FROM eshop_advanced_feature__virtual_category_in_group") as $row) {
			if ($row['id']) {
				$inGroup[] = $row['id'];
			}
		}

		$vcInGroups = [];
		foreach (array_chunk($inGroup, 200) as $ids) {
			foreach ($conn->fetchAllAssociative("SELECT * FROM eshop_advanced_feature__virtual_category_text WHERE id IN (" . implode(',', $ids) . ")") as $row) {
				$vcInGroups[$row['id']][$row['locale']] = $row;
			}
		}

		$langs = $this->langs->getLangs(false);
		foreach ($vcInGroups as $vcId => $texts) {
			$csText = $texts['cs'] ?? null;
			if (!$csText) {
				continue;
			}

			foreach ($langs as $lang) {
				if ($lang->getTag() === 'cs' || isset($texts[$lang->getTag()])) {
					continue;
				}

				$row = $csText;
				$conn->insert('eshop_advanced_feature__virtual_category_text', [
					'id'               => $row['id'],
					'locale'           => $lang->getTag(),
					'url'              => $row['url'],
					'page_title'       => $row['page_title'],
					'filtering_title'  => $row['filtering_title'],
					'page_description' => $row['page_description'],
					'h1'               => $row['h1'],
					'menu_title'       => $row['menu_title'],
					'description'      => $row['description'],
					'long_description' => $row['long_description'],
				]);
			}
		}
	}
}
