<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model\Navigation;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategories;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategoryGroups;
use EshopAdvancedFeature\Model\VirtualCategories as FrontVirtualCategories;
use Navigations\Model\NavigationConfig;
use Navigations\Model\Providers\INavigationItem;
use Nette\Utils\ArrayHash;
use Pages\AdminModule\Model\TemplatePages;

class VirtualCategoryGroup extends BaseNavigation implements INavigationItem
{
	public const COMPONENT_TYPE = 'eshopAdvancedFeature.navigation.virtualCategoryGroup';

	protected $title = 'group';

	protected $presenter = 'EshopAdvancedFeature:Front:Default';
	protected $action    = 'virtualCategoryGroup';

	protected VirtualCategoryGroups  $groupService;
	protected VirtualCategories      $virtualCategories;
	protected FrontVirtualCategories $frontVirtualCategories;
	protected TemplatePages          $templatePagesService;

	public function __construct(
		VirtualCategoryGroups  $groupService,
		VirtualCategories      $virtualCategories,
		TemplatePages          $templatePages,
		FrontVirtualCategories $frontVirtualCategories
	)
	{
		$this->groupService           = $groupService;
		$this->virtualCategories      = $virtualCategories;
		$this->templatePagesService   = $templatePages;
		$this->frontVirtualCategories = $frontVirtualCategories;
	}

	/**
	 * @inheritdoc
	 */
	public function getFormContainer(BaseForm $baseForm): BaseContainer
	{
		$container = new BaseContainer();

		$container->addSelect('group', $this->t('eshopAdvancedFeature.navigationBuilder.group'), $this->groupService->getOptionsForSelect())
			->setTranslator(null);

		$container->addSelect('templatePage', $this->t('eshopAdvancedFeature.navigationBuilder.template'), $this->templatePagesService->getOptionsForSelectGrouped())
			->setTranslator(null);

		return $container;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		return true;
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action): bool
	{
		return $presenter === $this->presenter && $action === $this->action;
	}

	/**
	 * @param DaoNavigationItem $navigation
	 * @param array             $urlParams
	 */
	public function routerIn($navigation, $urlParams = []): ?array
	{
		return [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => $navigation->componentParams['group'],
		];
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		unset($urlParams['productsFilter-filter']);
		$params = $navigation->getParam('virtualCategoryData');

		return $params['url'] ?? '/' . $navigation->alias;
	}

	/**
	 * @inheritdoc
	 */
	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$arr = [];

		$group = $this->groupService->get((int) $navigation->componentParams['group']);
		if (!$group) {
			return [];
		}

		$navs = $this->frontVirtualCategories->findNavigationsByGroups([$group->getId()]);

		if (!is_array($navs[$group->getId()])) {
			return [];
		}

		$category = $this->frontVirtualCategories->createCategoryDaoFromArray($navs[$group->getId()], $navigation);
		foreach ($category->getChild() as $child) {
			$link = $child->link;
			if (NavigationConfig::load('useEndSlash')) {
				$link = rtrim($link, '/') . '/';
			}

			$data = [
				'linkPath' => $link,
				'locale'   => $lang,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl;
				$seo        = [
					'robots'       => 'index, follow',
					'addToSiteMap' => 1,
				];
				$siteMapUrl->setSeo($seo);
				$data['siteMapData']['default'] = $siteMapUrl;
			}
			$arr[] = $data;
		}

		return $arr;
	}
}
