<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model\Subscribers;

use Contributte\Events\Extra\Event\Application\ShutdownEvent;
use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use EshopAdvancedFeature\Model\EshopAdvancedFeatureConfig;
use EshopAdvancedFeature\Model\Helpers\VirtualCategoryHelper;
use EshopAdvancedFeature\Model\Publishers\VirtualCategoryPublisher;
use EshopAdvancedFeature\Model\VirtualCategories as DefaultVirtualCategories;
use Exception;
use Nette\DI\Container;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;
use Tracy\ILogger;

class ApplicationSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator   $em;
	protected Translator               $translator;
	protected DefaultVirtualCategories $virtualCategories;
	protected Sites                    $sites;
	protected Container                $container;

	public function __construct(
		EntityManagerDecorator   $em,
		Translator               $translator,
		DefaultVirtualCategories $virtualCategories,
		Sites                    $sites,
		Container                $container
	)
	{
		$this->em                = $em;
		$this->translator        = $translator;
		$this->virtualCategories = $virtualCategories;
		$this->sites             = $sites;
		$this->container         = $container;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			ShutdownEvent::class => 'shutdown',
		];
	}

	public function shutdown(ShutdownEvent $event): void
	{
		if (VirtualCategoryHelper::getChangeLastVisit() && EshopAdvancedFeatureConfig::load('allowUpdateLastVisit')) {
			foreach (VirtualCategoryHelper::getChangeLastVisit() as $row) {
				$this->em->getConnection()->executeQuery("UPDATE eshop_advanced_feature__virtual_category SET `last_visit` = ? WHERE `id` = ?", [
					(new DateTime())->format('Y-m-d H:i:s'), $row['id'],
				]);
			}
		}

		/** @var VirtualCategoryPublisher|null $virtualCategoryPublisher */
		$virtualCategoryPublisher = $this->container->hasService('virtualCategoryPublisher')
			? $this->container->getService('virtualCategoryPublisher')
			: null;

		if ($virtualCategoryPublisher) {
			try {
				$this->updateByRabbit($virtualCategoryPublisher);
			} catch (Exception $e) {
				Debugger::log($e, ILogger::EXCEPTION);
				$this->updateByDefault();
			}
		} else {
			$this->updateByDefault();
		}
	}

	protected function updateByRabbit(VirtualCategoryPublisher $virtualCategoryPublisher): void
	{
		if (!empty(VirtualCategoryHelper::getPreparedVirtualCategories())) {
			foreach (VirtualCategoryHelper::getPreparedVirtualCategories() as $key => $data) {
				$tmp = explode(':', $key, 2);
				$virtualCategoryPublisher->checkForNew([
					'siteIdent' => $tmp[0] ?? $this->sites->getCurrentSite()->getIdent(),
					'data'      => $data,
					'locale'    => $tmp[1] ?? $this->translator->getLocale(),
				]);
			}
		}
	}

	protected function updateByDefault(): void
	{
		if (!empty(VirtualCategoryHelper::getPreparedVirtualCategories())) {
			foreach (VirtualCategoryHelper::getPreparedVirtualCategories() as $key => $data) {
				$tmp = explode(':', $key, 2);

				$this->virtualCategories->checkForNew(
					$data,
					$tmp[0] ?? $this->sites->getCurrentSite()->getIdent(),
					$tmp[1] ?? $this->translator->getLocale()
				);
			}
		}
	}
}
