<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\Sites;
use Doctrine\ORM\Query\Expr\Join;
use EshopAdvancedFeature\Model\Entities\VirtualCategory;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryGroup;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryGroupFilter;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryInGroup;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryRelated;
use EshopAdvancedFeature\Model\Helpers\VirtualCategoryHelper;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopCatalog\FrontModule\Model\Features;
use EshopCatalog\Model\Config;
use Nette\Caching\Cache;
use Nette\Utils\DateTime;
use Nette\Utils\Json;
use Tracy\Debugger;

class VirtualCategories extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = VirtualCategory::class;

	protected Sites                  $sites;
	protected Features               $features;
	protected VirtualCategoriesCache $virtualCategoriesCache;
	protected GroupsCache            $groupsCache;
	protected Categories             $categories;

	protected array  $prepareCreateVirtualCategory = [];
	protected array  $cNavByGroups                 = [];
	protected ?array $cRelated                     = null;
	protected ?array $cGroups                      = null;
	protected ?array $cVirtualCategoryGroup        = null;

	public function __construct(
		Sites                  $sites,
		Features               $features,
		VirtualCategoriesCache $virtualCategoriesCache,
		GroupsCache            $groupsCache,
		Categories             $categories
	)
	{
		$this->sites                  = $sites;
		$this->features               = $features;
		$this->virtualCategoriesCache = $virtualCategoriesCache;
		$this->groupsCache            = $groupsCache;
		$this->categories             = $categories;
	}

	public function getNavigationDataByUrl(string $url, ?string $locale = null): array
	{
		return $this->getNavigationDataByUrlMultiple([$url], $locale)[$url] ?? [];
	}

	public function validateRelationData(array $data): ?array
	{
		foreach ($data as $k1 => $row) {
			foreach ($row as $k2 => $v) {
				if (!is_numeric($v)) {
					return null;
				}

				$data[$k1][$k2] = (int) $v;
			}

			if ($data[$k1]) {
				sort($data[$k1]);
			}
		}

		return $data;
	}

	public function createRelationHash(string $siteIdent, array $data): string
	{
		if ($data[VirtualCategory::keyCategories]) {
			sort($data[VirtualCategory::keyCategories]);
		} else {
			$data[VirtualCategory::keyCategories] = [];
		}

		if ($data[VirtualCategory::keyManufacturers]) {
			sort($data[VirtualCategory::keyManufacturers]);
		} else {
			$data[VirtualCategory::keyManufacturers] = [];
		}

		if ($data[VirtualCategory::keyFeatureValues]) {
			sort($data[VirtualCategory::keyFeatureValues]);
		} else {
			$data[VirtualCategory::keyFeatureValues] = [];
		}

		return md5($siteIdent . '/' . Json::encode([
				VirtualCategory::keyCategories    => $data[VirtualCategory::keyCategories],
				VirtualCategory::keyManufacturers => $data[VirtualCategory::keyManufacturers],
				VirtualCategory::keyFeatureValues => $data[VirtualCategory::keyFeatureValues],
			]));
	}

	public function getNavigationDataByRelations(string $siteIdent, array $data): array
	{
		$conn   = $this->em->getConnection();
		$result = [];
		$hashes = [];

		foreach ($data as $url => $row) {
			$row = $this->validateRelationData($row);

			$hashes[$this->createRelationHash($siteIdent, $row)] = $url;
		}

		foreach ($conn->fetchAllAssociative("SELECT vc.id, vc.relation_hash FROM eshop_advanced_feature__virtual_category vc WHERE vc.relation_hash IN ('" . implode("', '", array_keys($hashes)) . "')") as $row) {
			$result[$hashes[$row['relation_hash']]] = $row['id'];
		}

		return $result;
	}

	public function getUrlByHash(string $locale, string $hash): array
	{
		return $this->em->getConnection()->fetchAssociative("SELECT vc.id, vct.url FROM `eshop_advanced_feature__virtual_category` vc
						INNER JOIN eshop_advanced_feature__virtual_category_text vct ON vc.id = vct.id and vct.locale = :locale
						WHERE vc.relation_hash = :hash
						LIMIT 1", [
			'locale' => $locale,
			'hash'   => $hash,
		]) ?: [];
	}

	public function updateHash(int $id, string $hash): void
	{
		$this->em->getConnection()->update('eshop_advanced_feature__virtual_category', [
			'relation_hash' => $hash,
		], [
			'id' => $id,
		]);
	}

	public function removeRow(int $id): void
	{
		$this->em->getConnection()->delete('eshop_advanced_feature__virtual_category', [
			'id' => $id,
		]);
	}

	public function getNavigationsIdByUrlMultiple(string $siteIdent, array $urls, ?string $locale = null): array
	{
		$locale = $locale ?: $this->translator->getLocale();

		$urlQuery = count($urls) === 1
			? '= \'' . $urls[0] . '\''
			: 'IN (\'' . implode('\',\'', $urls) . '\')';

		$conn   = $this->em->getConnection();
		$result = [];

		foreach ($conn->fetchAllAssociative("SELECT vct.id, vct.url
		FROM eshop_advanced_feature__virtual_category_text vct
		INNER JOIN eshop_advanced_feature__virtual_category vc ON vct.id = vc.id AND vc.site_ident = :siteIdent
		WHERE vct.locale = :locale AND vct.url {$urlQuery}", [
			'locale'    => $locale,
			'siteIdent' => $siteIdent,
		]) as $virtualCategory) {
			$result[$virtualCategory['url']] = $virtualCategory['id'];
		}

		return $result;
	}

	public function getNavigationDataByUrlMultiple(array $urls, ?string $locale = null): array
	{
		$locale    = $locale ?: $this->translator->getLocale();
		$siteIdent = $this->sites->getCurrentSite()->getIdent();

		$urls = array_diff($urls, $this->categories->getAllCategoriesLinksWithoutEndSlash($siteIdent, $locale));

		if (empty($urls)) {
			return [];
		}

		$urlQuery = count($urls) === 1
			? '= \'' . $urls[0] . '\''
			: 'IN (\'' . implode('\',\'', $urls) . '\')';

		$conn   = $this->em->getConnection();
		$result = [];

		$texts = [];
		foreach ($conn->fetchAllAssociative("SELECT vc.icon, vc.relation_hash, vc.site_ident, vc.is_locked, vc.params, 
    	vct.id, vct.url, vct.h1, vct.description, vct.long_description, vct.page_title, vct.menu_title, vct.page_description, vc.last_visit, vct.texts
		FROM eshop_advanced_feature__virtual_category_text vct
		INNER JOIN eshop_advanced_feature__virtual_category vc ON vct.id = vc.id AND vc.site_ident = :siteIdent
		WHERE vct.locale = :locale AND vct.url {$urlQuery}", [
			'locale'    => $locale,
			'siteIdent' => $this->sites->getCurrentSite()->getIdent(),
		]) as $virtualCategory) {
			try {
				if ($virtualCategory['texts']) {
					$virtualCategory['texts'] = deserialization($virtualCategory['texts']);
				}
			} catch (\Exception $e) {
			}
			$texts[$virtualCategory['id']] = $virtualCategory;
		}

		if (!$texts) {
			return [];
		}

		$inQuery = count($texts) === 1
			? '= ' . key($texts)
			: 'IN (' . implode(',', array_keys($texts)) . ')';

		foreach ($conn->fetchAllAssociative("SELECT vc.id,
    		GROUP_CONCAT(DISTINCT cat.category_id) as cats, 
    		GROUP_CONCAT(DISTINCT manu.manufacturer_id) as manu, 
    		GROUP_CONCAT(DISTINCT f.feature_value_id) as features
		FROM eshop_advanced_feature__virtual_category vc
		LEFT JOIN eshop_advanced_feature__virtual_category_categories cat ON cat.virtual_category_id = vc.id
		LEFT JOIN eshop_advanced_feature__virtual_category_manufacturers manu ON manu.virtual_category_id = vc.id
		LEFT JOIN eshop_advanced_feature__virtual_category_features f ON f.virtual_category_id = vc.id
		WHERE vc.id {$inQuery}
		GROUP BY vc.id") as $virtualCategory) {
			$virtualCategory += $texts[$virtualCategory['id']];
			if (Strings::contains((string) $virtualCategory['cats'], ',')) {
				$virtualCategory['cats'] = explode(',', (string) $virtualCategory['cats']);
			}

			if ($virtualCategory['manu']) {
				$virtualCategory['manu'] = explode(',', (string) $virtualCategory['manu']);
			} else {
				unset($virtualCategory['manu']);
			}

			if ($virtualCategory['features']) {
				$features = [];

				foreach (explode(',', (string) $virtualCategory['features']) as $v) {
					$v = $this->features->getFeatureValueById((int) $v);

					if ($v) {
						$features[$v['featureId']][] = (int) $v['id'];
					}
				}

				$virtualCategory['features'] = array_map(static fn($v) => implode('|', $v), $features);
			} else {
				unset($virtualCategory['features']);
			}

			if (Config::load('allowRelatedCategories', false)) {
				$virtualCategory['related'] = $this->getRelated()[$virtualCategory['id']];
			}

			$virtualCategory['groups'] = $this->getGroupsByVirtualCategory()[$virtualCategory['id']] ?? [];

			$result[$virtualCategory['url']] = $virtualCategory;
		}

		return $result;
	}

	protected function getRelated(): array
	{
		if ($this->cRelated === null) {
			$this->cRelated = $this->virtualCategoriesCache->getCache()->load('related', function(&$dep) {
				$dep = [
					Cache::EXPIRATION => '1 month',
					Cache::Tags       => [VirtualCategoriesCache::cacheNamespace],
				];

				$result = [];

				foreach ($this->em->createQueryBuilder()->select('IDENTITY(vcr.virtualCategory) as virtualCategoryId, vcr.targetId, vcr.targetKey')
					         ->from(VirtualCategoryRelated::class, 'vcr')
					         ->orderBy('vcr.virtualCategory', 'ASC')->addOrderBy('vcr.position', 'ASC')
					         ->getQuery()->getArrayResult() as $row) {
					$result[$row['virtualCategoryId']][$row['targetId']] = $row;
				}

				return $result;
			});
		}

		return $this->cRelated;
	}

	public function findNavigationsByGroups(array $groups): array
	{
		$result   = [];
		$keys     = [];
		$whereIds = [];
		$locale   = $this->translator->getLocale();

		if (!$groups) {
			return $result;
		}

		foreach ($groups as $gId) {
			if (isset($this->cNavByGroups[$gId])) {
				$result[$gId] = $this->cNavByGroups[$gId];
			} else {
				$keys[] = 'navByGroups/' . $locale . '/' . $gId;
			}
		}

		if (!empty($keys)) {
			foreach ($this->virtualCategoriesCache->getCache()->bulkLoad($keys) as $key => $row) {
				$tmp = explode('/', $key);
				$gId = end($tmp);

				if ($row) {
					$result[$gId]             = $row;
					$this->cNavByGroups[$gId] = $row;
				} else {
					$whereIds[]   = $gId;
					$result[$gId] = null;
				}
			}
		}

		if (!empty($whereIds)) {
			foreach ($this->getEr()->createQueryBuilder('n')
				         ->select('n.id, nt.h1, nt.url, GROUP_CONCAT(IDENTITY(g.group)) as groups, n.icon, nt.menuTitle, nt.texts')
				         ->innerJoin('n.texts', 'nt', Join::WITH, 'nt.locale = :lang')
				         ->innerJoin('n.inGroups', 'g', Join::WITH, 'g.group IN (:groups)')
				         ->setParameters([
					         'lang'   => $locale,
					         'groups' => $whereIds,
				         ])
				         ->groupBy('n.id')
				         ->orderBy('g.group')->addOrderBy('g.position', 'asc')
				         ->getQuery()->getArrayResult() as $row) {
				if (!$row['texts']) {
					$result['texts'] = [];
				}

				$rowGroups = explode(',', $row['groups']);
				foreach ($rowGroups as $group) {
					$result[$group][$row['id']] = $row;
				}

				foreach ($rowGroups as $group) {
					$cacheDep = [
						Cache::Tags       => ['navByGroups'],
						Cache::EXPIRATION => '1 month',
					];
					$this->virtualCategoriesCache->getCache()->save('navByGroups/' . $locale . '/' . $group, $result[$group], $cacheDep);
				}
			}
		}

		return $result;
	}

	public function createCategoryDaoFromArray(array $data, DaoNavigationItem $nav): Category
	{
		$childs = [];

		foreach ($data as $row) {
			$dao               = new Category;
			$dao->id           = $row['id'];
			$dao->name         = $row['menuTitle'] ?: $row['h1'];
			$dao->image        = $row['icon'];
			$dao->defaultImage = $row['icon'];
			$dao->link         = '/' . ltrim($row['url'], '/');
			$dao->setAttrs([
				'texts' => $row['texts'],
			]);

			$childs[$row['id']] = $dao;
		}

		$category               = new Category;
		$category->id           = $nav->getId();
		$category->name         = $nav->title;
		$category->link         = is_array($nav->link) ? Json::encode($nav->link) : (string) $nav->link;
		$category->image        = $nav->ico;
		$category->defaultImage = $nav->ico;

		$category->setChild($childs);

		return $category;
	}

	public function checkForNew(array $prepared, ?string $siteIdent = null, ?string $locale = null): void
	{
		if (empty($prepared)) {
			return;
		}

		$siteIdent = $siteIdent ?: $this->sites->getCurrentSite()->getIdent();
		$locale    = $locale ?: $this->translator->getLocale();

		Sites::$currentIdentOverride = $siteIdent;
		Sites::$currentLangOverride  = $locale;
		$this->translator->setLocale($locale);

		$site   = $this->sites->getSites()[$siteIdent];
		$domain = $site->getDomains()[$locale] ?? null;

		if (!$domain || !$domain->isActive) {
			return;
		}

		Sites::$currentIdentOverride = $siteIdent;
		Sites::$currentLangOverride  = $locale;

		$this->processNew($prepared, $siteIdent, $locale);
	}

	public function processNew(array $prepared, string $siteIdent, string $locale): void
	{
		$exists = $this->getNavigationsIdByUrlMultiple($siteIdent, array_keys($prepared), $locale);

		$missing = array_diff_key($prepared, $exists, array_flip($this->categories->getAllCategoriesLinksWithoutEndSlash($siteIdent, $locale)));

		if (empty($missing)) {
			return;
		}

		// Vyhledani bez relaci bez jazyka
		$missingRelationData = array_diff_key($missing, $exists);
		$byRelations         = $this->getNavigationDataByRelations($siteIdent, $missingRelationData);

		if (!empty($byRelations)) {
			$insert = [];

			// Pokud existuje s jinou url tak tu ulozit do cache
			$byRelationsFlip = array_flip($byRelations);
			foreach ($this->em->getConnection()->fetchAllAssociative("SELECT vct.id, vct.url, vct.locale
					FROM eshop_advanced_feature__virtual_category_text vct
					WHERE vct.id IN (" . implode(',', $byRelations) . ") 
						AND vct.locale = ?", [
				$locale,
			]) as $row) {
				if ($missingRelationData[$byRelationsFlip[$row['id']]]) {
					//						if ($byRelationsFlip[$row['id']] && $row['url'] !== $byRelationsFlip[$row['id']]) {
					//							Debugger::log('', 'virtualCategoryTextUpdate');
					//							Debugger::log('OLD - ' . $row['url'], 'virtualCategoryTextUpdate');
					//							Debugger::log('NEW - ' . $byRelationsFlip[$row['id']], 'virtualCategoryTextUpdate');
					//							$row['url'] = $byRelationsFlip[$row['id']];
					//							try {
					//								$this->em->getConnection()->executeQuery("UPDATE eshop_advanced_feature__virtual_category_text vct SET vct.url = :url WHERE vct.id = :id AND vct.locale = :locale", [
					//									'url'    => $row['url'],
					//									'id'     => $row['id'],
					//									'locale' => $locale,
					//								]);
					//							} catch (\Exception $e) {
					//								Debugger::log($e->getMessage(), 'virtualCategoryTextUpdate');
					//							}
					//						}
					$this->virtualCategoriesCache->getCache()->save(VirtualCategoryHelper::getUrlCacheKey(
						$siteIdent,
						$locale,
						$this->validateRelationData($missingRelationData[$byRelationsFlip[$row['id']]]),
					), $row['url'], [
						Cache::EXPIRATION => '1 week',
					]);

					unset(
						$missing[$byRelationsFlip[$row['id']]],
						$byRelations[$byRelationsFlip[$row['id']]]
					);
				}
			}

			// Vlozeni jazyku
			foreach ($byRelations as $url => $id) {
				$insert[] = sprintf("(%u, '%s', '%s')", $id, $locale, $url);
			}

			if (!empty($insert)) {
				$this->em->getConnection()->executeQuery("INSERT IGNORE INTO eshop_advanced_feature__virtual_category_text (id, locale, url)
					VALUES " . implode(', ', $insert));
			}

			$missing = array_diff_key($missing, $byRelations);
		}

		// Vlozeni zakladnich dat
		if (!empty($missing)) {
		}

		foreach ($missing as $url => $data) {
			try {
				$data = $this->validateRelationData($data);
				if ($data === null) {
					continue;
				}

				$conn = $this->em->getConnection();

				$this->em->beginTransaction();

				$conn->executeQuery("INSERT INTO eshop_advanced_feature__virtual_category 
    						(created, site_ident, relation_hash)
							VALUES (:created, :siteIdent, :relationHash)", [
					'created'      => (new DateTime())->format('Y-m-d H:i:s'),
					'siteIdent'    => $siteIdent,
					'relationHash' => $this->createRelationHash($siteIdent, $data),
				]);

				$virtualCategoryId = $conn->lastInsertId();

				$conn->executeQuery("INSERT INTO eshop_advanced_feature__virtual_category_text 
    						(id, locale, url)
							VALUES (:id, :locale, :url)", [
					'id'     => $virtualCategoryId,
					'locale' => $locale,
					'url'    => $url,
				]);

				foreach ($data[VirtualCategory::keyCategories] as $v) {
					$conn->executeQuery("INSERT IGNORE INTO eshop_advanced_feature__virtual_category_categories (virtual_category_id, category_id) VALUES (:vcId, :cId)", [
						'vcId' => $virtualCategoryId,
						'cId'  => $v,
					]);
				}

				foreach ($data[VirtualCategory::keyManufacturers] as $v) {
					$conn->executeQuery("INSERT IGNORE INTO eshop_advanced_feature__virtual_category_manufacturers (virtual_category_id, manufacturer_id) VALUES (:vcId, :mId)", [
						'vcId' => $virtualCategoryId,
						'mId'  => $v,
					]);
				}

				foreach ($data[VirtualCategory::keyFeatureValues] as $v) {
					$conn->executeQuery("INSERT IGNORE INTO eshop_advanced_feature__virtual_category_features (virtual_category_id, feature_value_id) VALUES (:vcId, :fId)", [
						'vcId' => $virtualCategoryId,
						'fId'  => $v,
					]);
				}

				$this->em->commit();
			} catch (\Exception $e) {
				if ($this->em->getConnection()->isTransactionActive()) {
					$this->em->rollback();
				}

				Debugger::log($url, 'virtualCategoryCreateError');
				Debugger::log(json_encode($data), 'virtualCategoryCreateError');
				Debugger::log($e, 'virtualCategoryCreateError');

				if ($e->getMessage() === 'The EntityManager is closed.') {
					break;
				}
			}
		}
	}

	public function prepareUrlsByRelation(?string $siteIdent = null, ?string $lang = null): void
	{
		$sites = [];
		if ($siteIdent) {
			$site = $this->sites->getSites(true)[$siteIdent] ?? null;
			if ($site) {
				$arr = [];

				if ($lang) {
					$arr[] = $lang;
				} else {
					foreach ($site->getDomains() as $domain) {
						$arr[] = $domain->getLang();
					}
				}

				$sites[$site->getIdent()] = $arr;
			}
		} else {
			foreach ($this->sites->getSites(true) as $site) {
				$arr = [];

				if ($lang) {
					$arr[] = $lang;
				} else {
					foreach ($site->getDomains() as $domain) {
						$arr[] = $domain->getLang();
					}
				}

				$sites[$site->getIdent()] = $arr;
			}
		}

		foreach ($sites as $ident => $langs) {
			foreach ($langs as $identLang) {
				foreach ($this->em->getConnection()->executeQuery("SELECT vct.url, vct.locale, vc.site_ident,
	                    (SELECT GROUP_CONCAT(vcCat.category_id) FROM eshop_advanced_feature__virtual_category_categories vcCat WHERE vcCat.virtual_category_id = vc.id) as cats,
	                    (SELECT GROUP_CONCAT(vcManu.manufacturer_id) FROM eshop_advanced_feature__virtual_category_manufacturers vcManu WHERE vcManu.virtual_category_id = vc.id) as manu,
	                    (SELECT GROUP_CONCAT(vcFea.feature_value_id) FROM eshop_advanced_feature__virtual_category_features vcFea WHERE vcFea.virtual_category_id = vc.id) as fea
					FROM eshop_advanced_feature__virtual_category vc
				    INNER JOIN eshop_advanced_feature__virtual_category_text vct ON vct.id = vc.id AND vct.locale = :lang
					WHERE vc.site_ident = :ident
					ORDER BY vc.last_visit DESC LIMIT 50000", [
					'lang'  => $identLang,
					'ident' => $ident,
				])->iterateAssociative() as $row) {
					$relationData = [
						VirtualCategory::keyCategories    => $row['cats'] ? array_map(static fn($v) => (int) $v, explode(',', (string) $row['cats'])) : [],
						VirtualCategory::keyManufacturers => $row['manu'] ? array_map(static fn($v) => (int) $v, explode(',', (string) $row['manu'])) : [],
						VirtualCategory::keyFeatureValues => $row['fea'] ? array_map(static fn($v) => (int) $v, explode(',', (string) $row['fea'])) : [],
					];

					$relationData = $this->validateRelationData($relationData);

					$key = VirtualCategoryHelper::getUrlCacheKey($row['site_ident'], $row['locale'], $relationData);

					$this->virtualCategoriesCache->getCache()->save($key, $row['url'], [
						Cache::EXPIRATION => '1 week',
					]);
				}

				$this->em->clear();
			}
		}
	}

	public function getGroups(): array
	{
		if ($this->cGroups === null) {
			$this->cGroups = $this->groupsCache->getCache()->load('groupsBase/' . $this->translator->getLocale(), function(&$dep) {
				$dep = [
					Cache::EXPIRATION => '1 month',
					Cache::Tags       => [GroupsCache::cacheNamespace],
				];

				$result = [];

				foreach ($this->em->getRepository(VirtualCategoryGroup::class)->createQueryBuilder('g', 'g.id')
					         ->select('g.id, gt.title, g.templatePageItem')
					         ->innerJoin('g.texts', 'gt', Join::WITH, 'gt.lang = :lang')
					         ->setParameters([
						         'lang' => $this->translator->getLocale(),
					         ])
					         ->groupBy('g.id')
					         ->getQuery()->getArrayResult() as $row) {
					$row['features']    = [];
					$result[$row['id']] = $row;
				}

				foreach ($this->em->getRepository(VirtualCategoryGroupFilter::class)->createQueryBuilder('gf')
					         ->select('IDENTITY(gf.group) as group, IDENTITY(gf.feature) as feature')
					         ->orderBy('gf.position', 'ASC')
					         ->getQuery()->getArrayResult() as $row) {
					if (isset($this->cGroups[$row['group']])) {
						$result[$row['group']]['features'][] = $row['feature'];
					}
				}

				return $result;
			});
		}

		return $this->cGroups;
	}

	public function getGroupsByVirtualCategory(): array
	{
		if ($this->cVirtualCategoryGroup === null) {
			$groups = $this->getGroups();

			$this->cVirtualCategoryGroup = $this->groupsCache->getCache()->load('groupsBaseByVc/' . $this->translator->getLocale(), function(&$dep) use (&$groups) {
				$dep = [
					Cache::EXPIRATION => '1 month',
					Cache::Tags       => [GroupsCache::cacheNamespace],
				];

				$result = [];

				foreach ($this->em->getRepository(VirtualCategoryInGroup::class)->createQueryBuilder('vcig')
					         ->select('IDENTITY(vcig.group) as group, IDENTITY(vcig.virtualCategory) as virtualCategory')
					         ->orderBy('vcig.position', 'ASC')
					         ->getQuery()->getArrayResult() as $row) {
					$result[$row['virtualCategory']][$row['group']] = $groups[$row['group']];
				}

				return $result;
			});
		}

		return $this->cVirtualCategoryGroup;
	}
}
