<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\AdminModule\Components\Group;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Core\AdminModule\Model\Sites;
use Core\Model\TemplateReader\TemplateReader;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategoryGroups;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryGroup;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryGroupFilter;
use EshopAdvancedFeature\Model\Entities\VirtualCategoryGroupText;
use EshopAdvancedFeature\Model\GroupsCache;
use EshopAdvancedFeature\Model\VirtualCategoriesCache;
use EshopCatalog\Model\Entities\Feature;
use Exception;
use Nette\Application\Attributes\Persistent;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use Pages\Model\Paths;

class GroupForm extends BaseControl
{
	#[Persistent]
	public ?int $id = null;

	public ?VirtualCategoryGroup $group = null;

	public function __construct(
		protected VirtualCategoryGroups  $virtualCategoryGroups,
		protected GroupsCache            $groupsCache,
		protected Sites                  $sites,
		protected TemplateReader         $templateReader,
		public Paths                     $pathsService,
		protected VirtualCategoriesCache $virtualCategoriesCache,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$this->templateReader->setTemplatesDir($this->pathsService->getTemplatePagesDir());
		$this->templateReader->setTranslateKey('templatePage');
		$templates = [$this->t('default.default') => $this->templateReader->getTemplates('default')];

		foreach (array_keys($this->sites->getOptionsForSelect()) as $k) {
			$this->templateReader->setTemplatesDir($this->pathsService->getTemplatePagesDir($k));
			$templates[$k] = $this->templateReader->getTemplates($k);
		}

		$filters = [];
		foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')->select('f.id, ft.name')
			         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
			         ->andWhere('f.useAsFilter = 1')
			         ->setParameters(new ArrayCollection([new Parameter('lang', $this->translator->getLocale())]))->orderBy('f.position')
			         ->getQuery()->getArrayResult() as $row) {
			$filters[$row['id']] = $row['name'];
		}

		$titlePrefix = 'eshopAdvancedFeature.virtualCategoryGroup.';
		$form->addText('title', $titlePrefix . 'title')
			->setIsMultilanguage();
		$form->addEditor('description', $titlePrefix . 'description')->setHeight(300)
			->setDisableAutoP(false)
			->setIsMultilanguage();
		$form->addFilesManager('icon', $titlePrefix . 'icon');

		$form->addSelect('templatePageItem', $titlePrefix . 'templatePageItem', ['' => ''] + $templates);
		$form->addSelect('templatePageGroup', $titlePrefix . 'templatePageGroup', ['' => ''] + $templates);

		$form->addSortableCheckboxList('filters', $this->t('eshopCatalog.categoryForm.availableFilters'), $filters)->setTranslator(null);

		$form->addSaveCancelControl();

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var VirtualCategoryGroupText[] $texts */
			$texts           = [];
			$groupsCacheKeys = [];

			if ($this->id) {
				$group = $this->virtualCategoryGroups->get($this->id);

				foreach ($this->langsService->getLangs(false) as $l => $v) {
					$groupsCacheKeys[] = GroupsCache::getKey((int) $this->id, $l);
					$groupsCacheKeys[] = 'navByGroups/' . $l . '/' . $this->id;
				}

				if (!$group) {
					throw new \InvalidArgumentException();
				}

				$texts = $group->texts->toArray();
			} else {
				$group = new VirtualCategoryGroup();
			}

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l]) && $v['title']) {
					$texts[$l] = new VirtualCategoryGroupText($group, $l, $v['title']);
				}

				if (!$v['title'] || $v['title'] == '') {
					if (isset($texts[$l])) {
						$this->em->remove($texts[$l]);
					}

					unset($texts[$l]);
					continue;
				}

				$texts[$l]->title       = $v['title'];
				$texts[$l]->description = $v['description'] ?: null;

				$this->em->persist($texts[$l]);
				$group->texts->set($l, $texts[$l]);
			}

			$group->icon              = $values->icon;
			$group->templatePageItem  = $values->templatePageItem;
			$group->templatePageGroup = $values->templatePageGroup;

			// FILTRY
			$formFilters  = $values->filters;
			$groupFilters = $group->filters->toArray();

			// nové
			foreach (array_diff($formFilters, array_keys($groupFilters)) as $k => $v) {
				/** @var Feature $feature */
				$feature               = $this->em->getReference(Feature::class, $v);
				$groupFilter           = new VirtualCategoryGroupFilter($group, $feature);
				$groupFilter->position = $k;

				$this->em->persist($groupFilter);
				$group->filters->set($v, $groupFilter);
			}

			// odstranit
			foreach (array_diff(array_keys($groupFilters), $formFilters) as $v) {
				$groupFilter = $group->filters->get($v);
				if ($groupFilter) {
					$group->filters->remove($v);
					$this->em->remove($groupFilter);
				}
			}

			// seřadit
			foreach ($formFilters as $k => $v) {
				$groupFilter = $group->filters->get($v);
				if ($groupFilter) {
					$groupFilter->position = $k;
					$this->em->persist($groupFilter);
				}
			}

			$this->em->persist($group);
			$this->em->flush();

			foreach ($this->langsService->getLangs(false) as $lang) {
				$this->groupsCache->removeGroup($group->getId(), $lang->getTag());
			}

			foreach (array_unique($groupsCacheKeys) as $key) {
				$this->virtualCategoriesCache->getCache()->remove($key);
				$this->groupsCache->getCache()->remove($key);
			}

			$this->groupsCache->removeBase();

			$form->addCustomData('featureNavigationGroup', $group->getId());
			$this->presenter->flashMessageSuccess('default.saved');
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setVirtualCategoryGroup(int $id): void
	{
		$this->id = $id;
		$group    = $this->virtualCategoryGroups->get($id);

		if (!$group) {
			throw new InvalidArgumentException('');
		}

		$this->group = $group;

		$d = [
			'icon'              => $group->icon,
			'filters'           => $group->filters->getKeys(),
			'templatePageItem'  => $group->templatePageItem,
			'templatePageGroup' => $group->templatePageGroup,
		];

		foreach ($group->texts->toArray() as $lang => $vals) {
			$d['title'][$lang]       = $vals->title;
			$d['description'][$lang] = $vals->description;
		}

		$this['form']->setDefaults($d);
	}
}
