<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\AdminModule\Components\VirtualCategory;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\SaveCancelControl;
use Doctrine\ORM\Query\Expr\Join;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategories;
use EshopAdvancedFeature\AdminModule\Model\VirtualCategoryGroups;
use EshopAdvancedFeature\Model\Entities\VirtualCategory;
use EshopAdvancedFeature\Model\EshopAdvancedFeatureConfig;
use Nette\Application\UI\Multiplier;
use Nette\Utils\Html;

class VirtualCategoriesGrid extends BaseControl
{
	public array $filter = [];

	public function __construct(
		protected string                          $siteIdent,
		protected VirtualCategories               $virtualCategories,
		protected VirtualCategoryGroups           $virtualCategoryGroups,
		protected IVirtualCategoryFormFactory     $formFactory,
		protected IRelatedFormFactory             $relatedFormFactory,
		protected IVirtualCategoryTextFormFactory $textFormFactory,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ============= Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle       = $this->t('eshopAdvancedFeature.title.addVirtualCategory');
		$this->template->modal            = 'form';
		$this->template->modalDialogClass = 'modal-xxl';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this['form']->id                 = $id;
		$this->template->modalTitle       = $this->t('eshopAdvancedFeature.title.editVirtualCategory',
			['title' => $this['form']->getVirtualCategory()->getText()->h1]);
		$this->template->modal            = 'form';
		$this->template->modalDialogClass = 'modal-xxl';
		$this->redrawControl('modal');
	}

	public function handleTexts(int $id): void
	{
		$this['textForm']->id             = $id;
		$this->template->modalTitle       = $this->t('eshopAdvancedFeature.title.editVirtualCategoryTexts',
			['title' => $this['textForm']->getVirtualCategory()->getText()->h1]);
		$this->template->modal            = 'textForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	/**
	 * @param int|string|int[]|string[] $id
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->presenter;

		if (is_array($id)
			? $this->virtualCategories->removeMultiple($id)
			: $this->virtualCategories->remove($id)) {
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		$presenter->redrawControl('flashes');
		$this['grid']->reload();
	}

	/**
	 * @param int|string|array $id
	 */
	public function handleAddToSitemap($id): void
	{
		$presenter = $this->presenter;

		if (!is_array($id)) {
			$id = [$id];
		}

		$this->virtualCategories->addToSitemap($id);

		$presenter->redrawControl('flashes');
		$this['grid']->reload();
	}

	/**
	 * @param int|string|array $id
	 */
	public function handleRemoveFromSitemap($id): void
	{
		$presenter = $this->presenter;

		if (!is_array($id)) {
			$id = [$id];
		}

		$this->virtualCategories->removeFromSitemap($id);

		$presenter->redrawControl('flashes');
		$this['grid']->reload();
	}

	public function handleRelated(int $id): void
	{
		$virtualCategory = $this->virtualCategories->get($id);

		if ($virtualCategory) {
			$this->template->modalTitle       = $this->t('eshopAdvancedFeature.title.editRelated', ['title' => trim($virtualCategory->getText()->pageTitle . ' (' . $virtualCategory->getText()
					->getUrl() . ')')]);
			$this->template->modal            = 'relatedForm';
			$this->template->modalDialogClass = 'modal-xl';
			$this->template->id               = $id;
			$this->redrawControl('modal');
		}
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ============= Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$columnPrefix = 'eshopAdvancedFeature.virtualCategory.';

		if (
			$this->filter === []
			|| (
				empty($this->filter['urlExact'])
				&& empty($this->filter['urlContain'])
				&& empty($this->filter['groups'])
				&& empty($this->filter['categories'])
				&& empty($this->filter['manufacturers'])
				&& empty($this->filter['featureValues']
				)
			)
		) {
			$grid->setDataSource([]);

			$grid->addColumnText('url', $columnPrefix . 'url', 'text.url');
			$grid->addColumnText('h1', $columnPrefix . 'h1', 'text.h1');
			$grid->addColumnText('group', $columnPrefix . 'groups');
			$grid->addColumnText('isLocked', $columnPrefix . 'isLocked');
			$grid->addColumnText('inSitemap', $columnPrefix . 'inSitemap');

			return $grid;
		}

		$qb = $this->virtualCategories->getEr()->createQueryBuilder('vc')
			->addSelect('t')
			->innerJoin('vc.texts', 't', Join::WITH, 't.locale = :locale')
			->andWhere('vc.siteIdent = :siteIdent')
			->setParameters(new ArrayCollection([new Parameter('locale', $this->translator->getLocale()), new Parameter('siteIdent', $this->siteIdent)]))
			->addOrderBy('vc.created', 'DESC');

		if ($this->filter['urlExact'] ?? null) {
			$qb->andWhere('t.url = :urlExact')
				->setParameter('urlExact', $this->filter['urlExact']);
		}

		if ($this->filter['urlContain'] ?? null) {
			$qb->andWhere('t.url LIKE :urlContain')
				->setParameter('urlContain', '%' . $this->filter['urlContain'] . '%');
		}

		if ($this->filter['groups'] ?? null) {
			$qb->innerJoin('vc.inGroups', 'inG', Join::WITH, 'inG.group IN (:group)')
				->innerJoin('inG.group', 'g')
				->innerJoin('g.texts', 'gText', Join::WITH, 'gText.lang = :locale')
				->addSelect('inG, g, gText')
				->setParameter('group', $this->filter['groups']);
		} else {
			$qb->leftJoin('vc.inGroups', 'inG')
				->leftJoin('inG.group', 'g')
				->leftJoin('g.texts', 'gText', Join::WITH, 'gText.lang = :locale')
				->addSelect('inG, g, gText');
		}

		if ($this->filter['categories'] ?? null) {
			$qb->innerJoin('vc.categories', 'c', Join::WITH, 'c.id IN (:categories)')
				->setParameter('categories', $this->filter['categories']);
		}

		if ($this->filter['manufacturers'] ?? null) {
			$qb->innerJoin('vc.manufacturers', 'm', Join::WITH, 'm.id IN (:manufacturers)')
				->setParameter('manufacturers', $this->filter['manufacturers']);
		}

		if ($this->filter['featureValues'] ?? null) {
			$features = [];
			$fValues  = [];

			foreach ($this->filter['featureValues'] as $v) {
				if (\str_starts_with((string) $v, 'f')) {
					$features[] = (int) substr((string) $v, 1);
				} else if (is_numeric($v)) {
					$fValues[] = (int) $v;
				}
			}

			if ($features !== []) {
				$qb->innerJoin('vc.featureValues', 'fv', Join::WITH, 'fv.feature IN (:featureValuesF)')
					->setParameter('featureValuesF', $features);
			}

			if ($fValues !== []) {
				$qb->innerJoin('vc.featureValues', 'fv', Join::WITH, 'fv.id IN (:featureValuesV)')
					->setParameter('featureValuesV', $fValues);
			}
		}

		$grid->setDataSource($qb);

		// Toolbar
		$grid->addToolbarButton('add!', 'default.add')
			->setIcon('plus')
			->setClass('ajax btn-success btn');

		// Columns
		$grid->addColumnText('url', $columnPrefix . 'url', 'text.url');
		$grid->addColumnText('h1', $columnPrefix . 'h1', 'text.h1');
		$grid->addColumnText('group', $columnPrefix . 'groups')->setRenderer(static function(VirtualCategory $row) {
			$html = Html::el();

			foreach ($row->inGroups as $g) {
				$html->addHtml(Html::el('div')->setText($g->group->getText()->title));
			}

			return $html;
		});
		$grid->addColumnText('isLocked', $columnPrefix . 'isLocked')
			->setRenderer(static function(VirtualCategory $row) {
				if ($row->isLocked()) {
					return Html::el('span class="btn btn-success"')
						->addHtml(Html::el('i class="fa fa-check"'));
				}

				return '';
			})->setAlign('center');
		$grid->addColumnText('inSitemap', $columnPrefix . 'inSitemap')
			->setRenderer(static function(VirtualCategory $row) {
				if ($row->addToSitemap !== 0) {
					return Html::el('span class="btn btn-success"')
						->addHtml(Html::el('i class="fa fa-check"'));
				}

				return '';
			})->setAlign('center');

		// Action
		$grid->addAction('edit', '', 'edit!', ['id' => 'id'])
			->setIcon('edit')
			->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('texts', '', 'texts!', ['id' => 'id'])
			->setIcon('font')
			->setBsType('primary')
			->addClass('ajax');
		if (EshopAdvancedFeatureConfig::load('allowRelated')) {
			$grid->addAction('related', '', 'related!')->setIcon('link')->setBsType('default')
				->addClass('ajax');
		}
		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		$grid->addGroupAction('default.delete')->onSelect[]                    = $this->handleDelete(...);
		$grid->addGroupAction($columnPrefix . 'addToSitemap')->onSelect[]      = $this->handleAddToSitemap(...);
		$grid->addGroupAction($columnPrefix . 'removeFromSitemap')->onSelect[] = $this->handleRemoveFromSitemap(...);

		// Prototype
		$grid->getColumn('group')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('group')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('isLocked')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('inSitemap')->getElementPrototype('th')->addClass('w1nw');

		return $grid;
	}

	protected function createComponentForm(): VirtualCategoryForm
	{
		$control = $this->formFactory->create($this->siteIdent);

		if ($this->getParameter('id')) {
			$control->setVirtualCategory((int) $this->getParameter('id'));
		}

		$control->onAnchor[] = function() use ($control): void {
			$control['form']->onSuccessSave[] = function(BaseForm $baseForm): void {
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
				$this->handleEdit((int) $baseForm->getCustomData('virtualCategoryId'));
			};

			$control['form']->onSuccessSaveAndClose[] = function(): void {
				$this->presenter->payload->hideModal = true;
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
			};

			/** @var SaveCancelControl $saveControl */
			$saveControl = $control['form']->getComponent('saveControl');
			$saveControl->closeModalOnCancel();
		};

		return $control;
	}

	protected function createComponentTextForm(): VirtualCategoryTextForm
	{
		$control = $this->textFormFactory->create();

		if ($this->getParameter('id')) {
			$control->setVirtualCategory((int) $this->getParameter('id'));
		}

		$control->onAnchor[] = function() use ($control): void {
			$control['form']->onSuccessSave[] = function(BaseForm $baseForm): void {
				$this->presenter->redrawControl('flashes');
			};

			$control['form']->onSuccessSaveAndClose[] = function(): void {
				$this->presenter->payload->hideModal = true;
				$this->presenter->redrawControl('flashes');
			};

			/** @var SaveCancelControl $saveControl */
			$saveControl = $control['form']->getComponent('saveControl');
			$saveControl->closeModalOnCancel();
		};

		return $control;
	}

	protected function createComponentRelatedForm(): Multiplier
	{
		return new Multiplier(fn($id): RelatedForm => $this->relatedFormFactory->create((int) $id));
	}
}
