<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\AdminModule\Model\Subscribers;

use Contributte\EventDispatcher\EventSubscriber;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Doctrine\ORM\EntityManager;
use EshopCatalog\AdminModule\Components\Features\FeatureForm;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\Feature;
use Nette\Caching\Cache;

class FeatureFormSubscriber implements EventSubscriber
{
	/** @var EntityManager */
	protected $entityManager;

	/** @var CacheService */
	protected $cacheService;

	public function __construct(EntityManager $entityManager, CacheService $cacheService)
	{
		$this->entityManager = $entityManager;
		$this->cacheService  = $cacheService;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			FeatureForm::class . '::createForm'  => ['createForm', 100],
			FeatureForm::class . '::formSuccess' => ['formSuccess', 100],
			FeatureForm::class . '::setFeature'  => ['setFeature', 100],
		];
	}

	public function createForm(CreateFormEvent $event): void
	{
		$form = $event->form;

		$form->addGroup('eshopAdvancedFeature.featureForm.title');
		$form->addBool('beautyUrlActive', 'eshopAdvancedFeature.feature.isActive')
			->setDefaultValue(0);
		$form->addBool('beaUrlPosition', 'eshopAdvancedFeature.feature.oneIsPrefix', [
			0 => 'eshopAdvancedFeature.feature.beforeTitle',
			1 => 'eshopAdvancedFeature.feature.afterTitle',
		]);
		$form->addText('beaUrlPriority', 'eshopAdvancedFeature.feature.prefixPriority')
			->setDefaultValue(10)->setType('number');
		$form->addText('beaUrlMulti', 'eshopAdvancedFeature.feature.multi')->setIsMultilanguage();
		$form->addBool('beaUrlPositionMulti', 'eshopAdvancedFeature.feature.multiIsPrefix', [
			0 => 'eshopAdvancedFeature.feature.beforeTitle',
			1 => 'eshopAdvancedFeature.feature.afterTitle',
		]);
		$form->addText('beaUrlPriorityMulti', 'eshopAdvancedFeature.feature.prefixPriorityMultiple')
			->setDefaultValue(10)->setType('number');
		$form->addHidden('beaUrlPattern', 'eshopAdvancedFeature.feature.pattern')->setDefaultValue('%v%');
	}

	public function formSuccess(FormSuccessEvent $event): void
	{
		/** @var Feature $feature */
		$feature = $event->custom['entity'];
		$values  = $event->values;

		$position      = $values->beaUrlPosition ? 'suffix' : 'prefix';
		$multiPosition = $values->beaUrlPositionMulti ? 'suffix' : 'prefix';

		$feature->beautyUrlActive = $values->beautyUrlActive;
		$feature->beaUrlPosition  = $position;
		$feature->beaUrlPriority  = $values->beaUrlPriority;
		$feature->beaUrlPattern   = $values->beaUrlPattern;

		$feature->beaUrlPositionMulti = $multiPosition;
		$feature->beaUrlPriorityMulti = $values->beaUrlPriorityMulti;
		$this->entityManager->persist($feature);

		foreach ($feature->getFeatureTexts() as $l => $text) {
			$text->beaUrlMulti = $values->beaUrlMulti[$l] ?? null;
			$this->entityManager->persist($text);;
		}

		$this->entityManager->flush();

		$this->cacheService->clean('navigation', [Cache::TAGS => ['categoriesFilter_ff_' . $feature->getId()]]);
	}

	public function setFeature(SetFormDataEvent $event): void
	{
		/** @var Feature $feature */
		$feature = $event->entity;

		if ($feature) {
			foreach ($feature->getFeatureTexts() as $l => $text)
				$multi[$l] = $text->beaUrlMulti;

			$event->form->setDefaults([
				'beautyUrlActive'     => $feature->beautyUrlActive ? 1 : 0,
				'beaUrlPosition'      => $feature->beaUrlPosition === 'prefix' ? 0 : 1,
				'beaUrlPriority'      => $feature->beaUrlPriority,
				'beaUrlPattern'       => $feature->beaUrlPattern,
				'beaUrlMulti'         => $multi,
				'beaUrlPositionMulti' => $feature->beaUrlPositionMulti === 'prefix' ? 0 : 1,
				'beaUrlPriorityMulti' => $feature->beaUrlPriorityMulti,
			]);
		}
	}
}
