<?php declare(strict_types = 1);

namespace EshopAdvancedFeature\Model;

use EshopCatalog\FrontModule\Model\Dao\Manufacturer;
use EshopCatalog\FrontModule\Model\Manufacturers;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use Kdyby\Doctrine\EntityManager;
use Nette\Localization\ITranslator;

class PrefixSuffixGenerator
{
	/** @var EntityManager */
	protected $entityManager;

	/** @var Manufacturers */
	protected $manufactures;

	/** @var ITranslator */
	protected $translator;

	/** @var array */
	protected $cFeatureValuesTexts, $cFeatureAdvanced, $cCategoryRody;

	public function __construct(EntityManager $em, Manufacturers $manufacturers, ITranslator $translator)
	{
		$this->entityManager = $em;
		$this->manufactures  = $manufacturers;
		$this->translator    = $translator;
	}

	public function generate(array $features, array $manufactures, array $priceRange, ?int $categoryId = null, ?string $text = null): array
	{
		$rod = $categoryId ? $this->getCategoryRod($categoryId) : null;

		$result = [
			'prefix'       => '',
			'suffix'       => '',
			'text'         => $text,
			'isMultiple'   => false,
			'priceRange'   => false,
			'usedFeatures' => [],
		];

		if (!empty($features)) {
			$featureIds = array_keys($features);
			$valueIds   = [];

			foreach ($features as $row) {
				$valueIds = array_merge($valueIds, $row);
			}
			$featuresR = [
				'prefix' => [],
				'suffix' => [],
			];
			$values    = $this->getFeatureValueTexts($this->translator->getLocale(), $valueIds);

			foreach ($this->getFeatureAdvanced($featureIds) as $row) {
				if (count($features[$row['id']]) > 1) {
					$result['isMultiple'] = true;
					$tmp                  = [
						'position' => $row['beaUrlPositionMulti'],
						'priority' => (int) $row['beaUrlPriorityMulti'],
						'text'     => $row['beaUrlMulti'],
					];
				} else {
					$featureRod  = $rod ? 'rod' . $rod : 'name';
					$featureText = $values[$features[$row['id']][0]][$featureRod] ?: $values[$features[$row['id']][0]]['name'];

					$tmp = [
						'position' => $row['beaUrlPosition'],
						'priority' => (int) $row['beaUrlPriority'],
						'text'     => trim(str_replace('%v%', $featureText, $row['beaUrlPattern'])),
					];
				}
				$result['usedFeatures'][]      = (int) $row['id'];
				$tmp['feature']                = (int) $row['id'];
				$featuresR[$tmp['position']][] = $tmp;
			}

			usort($featuresR['prefix'], function($a, $b) {
				return $a->priority <=> $b->priority;
			});

			foreach ($featuresR as $position => $group) {
				foreach ($group as $feature) {
					$result[$position] .= ' ' . $feature['text'];
				}
			}
		}

		if (!empty($manufactures)) {
			if (count($manufactures) > 1) {
				$result['prefix']     .= ' ' . $this->translator->translate('eshopAdvancedFeatureFront.moreManufacturers.' . strtolower($rod ?: 'z'));
				$result['isMultiple'] = true;
			} else {
				/** @var Manufacturer $manu */
				$manu = $this->manufactures->get((int) $manufactures[0]);
				if ($manu)
					$result['suffix'] .= ' ' . $manu->name;
			}
		}

		if (isset($priceRange['min']) || isset($priceRange['max']))
			$result['priceRange'] = true;
		if (isset($priceRange['min']))
			$result['suffix'] .= ' ' . $this->translator->translate('eshopAdvancedFeatureFront.priceFrom',
					['price' => number_format((float) $priceRange['min'], 0, '', ' ')]) . ' kč';
		if (isset($priceRange['max']))
			$result['suffix'] .= ' ' . $this->translator->translate('eshopAdvancedFeatureFront.priceTo',
					['price' => number_format((float) $priceRange['max'], 0, '', ' ')]) . ' kč';

		if ($text) {
			$result['text'] = $this->addPrefixSuffix($text, $result['prefix'], $result['suffix']);
		}

		return $result;
	}

	public function normalizeText(string $text): string
	{
		$expl = explode(' ', $text);
		$tmp  = substr($expl[0], 1);
		if ($tmp === strtolower($tmp))
			$text = str_replace($expl[0], strtolower($expl[0]), $text);

		return $text;
	}

	public function addPrefixSuffix(string $text, string $prefix = '', string $suffix = ''): string
	{
		return ucfirst(trim($prefix . ' ' . $this->normalizeText($text) . $suffix));
	}

	protected function getFeatureValueTexts(string $lang, array $valueIds): array
	{
		if (!isset($this->cFeatureValuesTexts[$lang])) {
			$this->cFeatureValuesTexts[$lang] = [];
			foreach ($this->entityManager->getRepository(FeatureValueTexts::class)->createQueryBuilder('fvt')
				         ->select('IDENTITY(fvt.id) as id, fvt.name, fvt.rodM, fvt.rodZ, fvt.rodS')
				         ->andWhere('fvt.lang = :lang')
				         ->setParameters(['lang' => $lang])->getQuery()->getArrayResult() as $row)
				$this->cFeatureValuesTexts[$lang][$row['id']] = $row;
		}

		$result = [];
		foreach ($this->cFeatureValuesTexts[$lang] as $k => $v) {
			if (in_array($k, $valueIds))
				$result[$k] = $v;
		}

		return $result;
	}

	protected function getFeatureAdvanced(array $featureIds): array
	{
		if ($this->cFeatureAdvanced === null) {
			$this->cFeatureAdvanced = [];
			foreach ($this->entityManager->getRepository(Feature::class)->createQueryBuilder('f')
				         ->select('f.id, f.beaUrlPosition, f.beaUrlPriority, f.beaUrlPattern, f.beaUrlPositionMulti, f.beaUrlPriorityMulti')
				         ->addSelect('ft.beaUrlMulti')
				         ->andWhere('f.beautyUrlActive = 1')
				         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')->setParameter('lang', $this->translator->getLocale())
				         ->orderBy('f.beaUrlPriority', 'ASC')->getQuery()->getArrayResult() as $row)
				$this->cFeatureAdvanced[$row['id']] = $row;
		}

		$result = [];
		foreach ($this->cFeatureAdvanced as $k => $v) {
			if (in_array($k, $featureIds))
				$result[$k] = $v;
		}

		return $result;
	}

	protected function getCategoryRod(int $id)
	{
		if ($this->cCategoryRody === null) {
			$this->cCategoryRody = [];
			foreach ($this->entityManager->getRepository(Category::class)->createQueryBuilder('c')
				         ->select('c.id, c.rod')
				         ->andWhere('c.rod IS NOT NULL')
				         ->getQuery()->getArrayResult() as $row)
				$this->cCategoryRody[$row['id']] = strtoupper($row['rod']);
		}

		return $this->cCategoryRody[$id] ?? 'z';
	}
}
