<?php declare(strict_types = 1);

namespace EshopBulkOrder\FrontModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopBulkOrder\Model\Config;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao\AddedCartItem;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\RemovedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use EshopCatalog\Model\Config as EshopCatalogConfig;
use Nette\Utils\ArrayHash;

class ProductsList extends BaseControl
{
	/** @var Config */
	protected $config;

	/** @var ProductsFacade */
	protected $productsFacade;

	/** @var Categories */
	protected $categories;

	/** @var Carts */
	protected $cartService;

	/** @var Product[] */
	protected $cProducts;

	public function __construct(Config $config, ProductsFacade $productsFacade, Carts $carts, Categories $categories)
	{
		$this->config      = $config;
		$this->productsFacade    = $productsFacade;
		$this->cartService = $carts;
		$this->categories  = $categories;
	}

	public function render(): void
	{
		$products  = $this->getProducts();
		$cartItems = $this->getCartItems();

		$totalPrice           = 0;
		$totalPriceWithoutVat = 0;

		foreach ($cartItems as $item) {
			$totalPrice           += $item->getTotalPrice();
			$totalPriceWithoutVat += $item->getTotalPriceWithoutVat();
		}

		$this->template->showGroupName        = Config::load('showGroupName', false);
		$this->template->totalPrice           = $totalPrice;
		$this->template->totalPriceWithoutVat = $totalPriceWithoutVat;
		$this->template->products             = $products;
		$this->template->cartItems            = $cartItems;
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$cartItems = $this->getCartItems();

		foreach ($this->getProducts() as $products) {
			foreach ($products as $product) {
				/** @var Product $product */
				$default = isset($cartItems[$product->getId()]) ? (int) $cartItems[$product->getId()]->quantity : 0;
				$input   = $form->addText('q_' . $product->getId(), '')->setDefaultValue($default)
					->setAttribute('data-add-to-cart-quantity-input', 'instant')
					->setAttribute('data-add-to-cart-allow-zero')
					->setAttribute('data-item-id', $product->getId())
					->setAttribute('min', 0)
					->setAttribute('data-min', 0);

				if (EshopCatalogConfig::load('pseudoWarehouse') && $product->unlimitedQuantity !== 1) {
					$input->setAttribute('max', $product->getQuantity());
					$input->setAttribute('data-max', $product->getQuantity());
				}
			}
		}

		$form->addSubmit('submit', 'eshopBulkOrderFront.bulkOrder.toCart');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		$products  = [];
		$cartItems = $this->getCartItems();

		foreach ($values as $k => $v) {
			$v         = (int) $v;
			$k         = explode('_', $k);
			$productId = (int) $k[1];

			$products[$productId] = $v;

			if (isset($cartItems[$productId])) {
				$item = new UpdatedCartItemEvent($cartItems[$productId]->getId(), $v);
				$this->eventDispatcher->dispatch('eshopOrders.cartUpdateItem', $item);
			} else if ($v > 0) {
				$addedCartItem            = new AddedCartItem();
				$addedCartItem->productId = $productId;
				$addedCartItem->quantity  = $v;
				$item                     = new AddedCartItemEvent($addedCartItem);
				$this->eventDispatcher->dispatch('eshopOrders.cartAddItem', $item);
			}
		}

		foreach (array_diff_key($cartItems, $products) as $v) {
			$item = new RemovedCartItemEvent($cartItems[$v]->getId());
			$this->eventDispatcher->dispatch('eshopOrders.cartRemoveItem', $item);
		}

		if (!is_bool($form->isSubmitted()))
			$this->getPresenter()->redirect(':EshopOrders:Front:Default:order');
	}

	protected function getProducts()
	{
		if ($this->cProducts === null) {
			$this->cProducts = [];
			$featureOthers   = [];
			$groupBy         = Config::load('groupBy');
			$query           = (new ProductQuery($this->translator->getLocale()))
				->withTexts()
				->selectIds();

			if (EshopCatalogConfig::load('onlyStockProductsInList', true) == true)
				$query->onlyInStockOrSupplier();

			$qb = $query->getQueryBuilder($this->productsFacade->productsService->getEr());
			$qb->innerJoin('p.idCategoryDefault', 'defCat');

			if ($groupBy === 'category') {
				$qb->orderBy('defCat.root')->addOrderBy('defCat.lft');
			} else if (isset($groupBy['feature'])) {
				foreach ($this->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')
					         ->select('fvt.name')->where('fv.feature = :feature')
					         ->innerJoin('fv.featureValueTexts', 'fvt', 'with', 'fvt.lang = :lang')
					         ->setParameters([
						         'feature' => $groupBy['feature'],
						         'lang'    => $this->translator->getLocale(),
					         ])
					         ->orderBy('fv.position')->getQuery()->getArrayResult() as $row) {
					$this->cProducts[$row['name']] = [];
				}
			}

			$ids = [];
			foreach ($qb->getQuery()->getArrayResult() as $row)
				$ids[] = $row['id'];

			foreach ($this->productsFacade->getProducts($ids) as $product) {
				$id = $product->getId();

				if ($groupBy === 'category') {
					$cat = $this->categories->get($product->getDefaultCategory());

					$this->cProducts[$cat->getNameH1()][$id] = $product;
				} else if (isset($groupBy['feature'])) {
					if (isset($product->getFeatures()[$groupBy['feature']]))
						$this->cProducts[$product->getFeatures()[$groupBy['feature']]->value][$id] = $product;
					else
						$featureOthers[$id] = $product;
				} else {
					$this->cProducts[''][$id] = $product;
				}
			}

			if (!empty($featureOthers))
				$this->cProducts[$this->translator->translate('eshopBulkOrderFront.groupBy.others')] = $featureOthers;
		}

		return $this->cProducts;
	}

	/**
	 * @return CartItem[]
	 */
	protected function getCartItems()
	{
		$cartItems = [];
		foreach ($this->cartService->getCurrentCart()->getCartItems() as $ci)
			$cartItems[$ci->getProductId()] = $ci;

		return $cartItems;
	}
}
