<?php declare(strict_types = 1);

namespace EshopBulkOrder\FrontModule\Components;

use Core\Model\Entities\QueryBuilder;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use EshopBulkOrder\Model\Config;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config as EshopCatalogConfig;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\Dao\AddedCartItem;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\RemovedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use EshopOrders\Model\Entities\Customer;
use Nette\Forms\Form;
use Nette\Utils\ArrayHash;
use Users\Model\Security\User;

class ProductsList extends BaseControl
{
	protected Config         $config;
	protected ProductsFacade $productsFacade;
	protected Categories     $categories;
	protected Carts          $cartService;
	protected Sites          $sites;
	protected Customers      $customers;
	protected User           $user;

	protected array $checkedCategories = [];

	/** @var array<string, Product[]>|null */
	protected $cProducts;

	protected ?Customer $customer = null;

	public function __construct(
		Config         $config,
		ProductsFacade $productsFacade,
		Carts          $carts,
		Categories     $categories,
		Sites          $sites,
		Customers      $customers,
		User           $user
	)
	{
		$this->config         = $config;
		$this->productsFacade = $productsFacade;
		$this->cartService    = $carts;
		$this->categories     = $categories;
		$this->sites          = $sites;
		$this->customers      = $customers;
		$this->user           = $user;

		$this->customer = $this->user->isLoggedIn()
			/** @phpstan-ignore-next-line */
			? $this->customers->getByUser($this->user->getIdentity())
			: null;
	}

	public function render(): void
	{
		$products  = $this->getProducts();
		$cartItems = $this->getCartItems();

		$totalPrice           = 0;
		$totalPriceWithoutVat = 0;

		foreach ($cartItems as $item) {
			$totalPrice           += $item->getTotalPrice();
			$totalPriceWithoutVat += $item->getTotalPriceWithoutVat();
		}

		$this->template->showGroupName        = Config::load('showGroupName', false);
		$this->template->totalPrice           = $totalPrice;
		$this->template->totalPriceWithoutVat = $totalPriceWithoutVat;
		$this->template->products             = $products;
		$this->template->cartItems            = $cartItems;
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$cartItems = $this->getCartItems();

		foreach ($this->getProducts() as $products) {
			foreach ($products as $product) {
				$minimumAmount = $product->minimumAmount;

				$default = isset($cartItems[$product->getId()]) ? (int) $cartItems[$product->getId()]->quantity : 0;
				$input   = $form->addText('q_' . $product->getId(), '')->setDefaultValue($default)
					->setHtmlAttribute('data-add-to-cart-quantity-input', 'instant')
					->setHtmlAttribute('data-allow-zero', 'true')
					->setHtmlAttribute('data-item-id', $product->getId());

				if ($minimumAmount) {
					$input->setHtmlAttribute('data-minimum-amount', $minimumAmount)
						->addCondition(Form::MIN, 1)
						->addRule(
							Form::MIN,
							'eshopCatalogFront.cart.minimumQuantity',
							$minimumAmount > 1 ? $minimumAmount : 0
						);
				}

				if (EshopCatalogConfig::load('pseudoWarehouse')
					&& !$product->unlimitedQuantity
					&& (!EshopCatalogConfig::load('allowWarehouseOverdraft') || !$product->getAvailability()->warehouseOverdraft)
				) {
					$input->setHtmlAttribute('max', $product->getQuantity());
					$input->setHtmlAttribute('data-max', $product->getQuantity());
				}
			}
		}

		$form->addSubmit('submit', 'eshopBulkOrderFront.bulkOrder.toCart');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{

		$products  = [];
		$cartItems = $this->getCartItems();

		/** @phpstan-ignore-next-line */
		$event = new FormSuccessEvent($form, $values, $this->getPresenter(false) ? $this->template : null, $this->getPresenter(false) ? $this->getPresenter() : null);
		if ($this->getPresenter(false)) {
			/** @phpstan-ignore-next-line */
			$event->presenter = $this->getPresenter();
		}
		$event->control = $this;
		$this->eventDispatcher->dispatch($event, self::class . '::formSuccess');

		foreach ($values as $k => $v) {
			$v         = (int) $v;
			$k         = explode('_', $k);
			$productId = (int) $k[1];

			$products[$productId] = $v;

			if (isset($cartItems[$productId])) {
				$cartItem = $cartItems[$productId];
				$this->eventDispatcher->dispatch(new UpdatedCartItemEvent(
					(string) $cartItem->getId(),
					(int) $v,
					(int) $cartItems[$values->itemId]->quantity,
				), 'eshopOrders.cartUpdateItem');
			} else if ($v > 0) {
				$addedCartItem            = new AddedCartItem();
				$addedCartItem->productId = $productId;
				$addedCartItem->quantity  = $v;
				$item                     = new AddedCartItemEvent($addedCartItem);
				$this->eventDispatcher->dispatch($item, 'eshopOrders.cartAddItem');
			}
		}

		/** @var int $val */
		foreach (array_diff_key($cartItems, $products) as $val) {
			if (!isset($cartItems[$val])) {
				continue;
			}
			$item = new RemovedCartItemEvent($cartItems[$val]->getId());
			$this->eventDispatcher->dispatch($item, 'eshopOrders.cartRemoveItem');
		}

		if (!is_bool($form->isSubmitted()))
			$this->getPresenter()->redirect(':EshopOrders:Front:Default:order');
	}

	/**
	 * @return array<string, Product[]>
	 */
	protected function getProducts(): array
	{
		if ($this->cProducts === null) {
			$this->cProducts       = [];
			$featureOthers         = [];
			$groupBy               = Config::load('groupBy');
			$disabledFeatureValues = Config::load('disabledFeatureValues');
			$query                 = (new ProductQuery($this->translator->getLocale()))
				->withTexts()
				->selectIds();

			/** @var QueryBuilder $qb */
			$qb = $query->getQueryBuilder($this->productsFacade->productsService->getEr());

			if (Config::load('showOnlyCanAddToCart') === true) {
				if (EshopCatalogConfig::load('onlyStockProductsInList', true) == true) {
					$query->onlyInStockOrSupplier();
				}

				if (in_array('av', $qb->getAllAliases())) {
					$qb->andWhere('av.canAddToCart = 1');
				} else {
					$qb->innerJoin('p.availability', 'av', Join::WITH, 'av.canAddToCart = 1');
				}
			}

			$baseQb = clone $qb;

			$qb->innerJoin('p.sites', 'site', Join::WITH, 'site.site = :site')
				->setParameter('site', $this->sites->getCurrentSite()->getIdent())
				->innerJoin('site.category', 'defCat');
			$qb->leftJoin('p.isVariant', 'variants')
				->addSelect('variants.variantId');

			$qb->addSelect('defCat.id as catId');

			if (Config::load('disabledCategories')) {
				$qb->andWhere('site.category NOT IN (:disabledCategories)')
					->setParameter('disabledCategories', Config::load('disabledCategories'));
			}

			if ($groupBy === 'category') {
				$qb->orderBy('defCat.root')->addOrderBy('defCat.lft');
			} else if (isset($groupBy['feature'])) {
				$qbFeature = $this->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')
					->select('fvt.name')->where('fv.feature = :feature')
					->innerJoin('fv.featureValueTexts', 'fvt', Join::WITH, 'fvt.lang = :lang')
					->setParameters([
						'feature' => $groupBy['feature'],
						'lang'    => $this->translator->getLocale(),
					]);

				if ($disabledFeatureValues) {
					$qbFeature->andWhere('fv.id NOT IN (' . implode(',', $disabledFeatureValues) . ')');
				}

				foreach ($qbFeature->orderBy('fv.position')->getQuery()->getArrayResult() as $row) {
					$this->cProducts[$row['name']] = [];
				}
			}

			$categories = $this->categories->getCategories($this->categories->getRootIdForSite($this->sites->getCurrentSite()->getIdent()));
			$ids        = [];
			$variantIds = [];
			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$category = $categories[$row['catId']] ?? null;

				if ($category && !$this->checkCategoryAccess($category)) {
					continue;
				}

				$ids[] = $row['id'];

				if ($row['variantId']) {
					$variantIds[$row['id']] = $row['variantId'];
				}
			}

			if ($variantIds) {
				$qb = clone $baseQb;
				$qb->innerJoin('p.isVariant', 'variant', Join::WITH, 'variant.variantId IN (:varIds)')
					->setParameter('varIds', array_values($variantIds))
					->select('p.id, variant.isDefault, variant.variantId');
				$variants = [];
				foreach ($qb->getQuery()->getArrayResult() as $row) {
					if ($row['isDefault']) {
						$variants[$row['variantId']]['default'] = $row['id'];
					} else {
						$variants[$row['variantId']]['ids'][] = $row['id'];
					}
				}

				foreach ($variants as $data) {
					if (!isset($data['ids'])) {
						continue;
					}

					$index = array_search($data['default'], $ids);
					if (is_numeric($index)) {
						array_splice($ids, $index + 1, 0, $data['ids']);
					}
				}
			}

			foreach ($this->productsFacade->getProducts(array_values($ids)) as $k => $product) {
				if (!$product->canAddToCart && Config::load('showOnlyCanAddToCart') === true) {
					continue;
				}

				if ($disabledFeatureValues && !empty(array_intersect($product->featureValuesIds, $disabledFeatureValues))) {
					continue;
				}

				$id = $product->getId();

				if ($groupBy === 'category') {
					$cat = $product->defaultCategory;

					if ($cat) {
						$this->cProducts[$cat->getNameH1()][$id] = $product;
					}
				} else if (isset($groupBy['feature'])) {
					$featuresById = $product->getFeatureById($groupBy['feature']) ?? null;
					if ($featuresById) {
						$this->cProducts[$featuresById->value][$id] = $product;
					} else {
						$featureOthers[$id] = $product;
					}
				} else {
					$this->cProducts[''][$id] = $product;
				}
			}

			if (!empty($featureOthers)) {
				$othersKey = $this->translator->translate('eshopBulkOrderFront.groupBy.others');
				if (!isset($this->cProducts[$othersKey])) {
					$this->cProducts[$othersKey] = [];
				}
				$this->cProducts[$othersKey] += $featureOthers;
			}
		}

		return $this->cProducts;
	}

	/**
	 * @return CartItem[]
	 */
	protected function getCartItems()
	{
		$cartItems = [];
		foreach ($this->cartService->getCurrentCart()->getCartItems() as $ci) {
			$cartItems[$ci->getProductId()] = $ci;
		}

		return $cartItems;
	}

	protected function checkCategoryAccess(Category $category): bool
	{
		if (!isset($this->checkedCategories[$category->getId()])) {
			$this->checkedCategories[$category->getId()] = $this->categories->checkCategoryRestrictionAccess($category->allowedCustomerGroups, $this->customer);
		}

		return $this->checkedCategories[$category->getId()];
	}
}
