<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Doctrine\Common\Collections\Criteria;
use EshopCatalog\AdminModule\Model\VatRates;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\ProductSupplier;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\ProductVariantSupplier;
use EshopCatalog\Model\Entities\VatRate;
use Kdyby\Doctrine\QueryBuilder;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryTexts;
use Nette\Utils\Html;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\AdminModule\Model\Products;

/**
 * Class ProductsGrid
 * @package EshopCatalog\AdminModule\Components\Products
 */
class ProductsGrid extends BaseControl
{
	/** @var \EshopCatalog\AdminModule\Model\Products */
	protected $productServices;

	/** @var array */
	protected $quantities;

	/** @var array */
	protected $pvsInStock;

	/** @var array */
	protected $productCategories;

	public function __construct(Products $products)
	{
		$this->productServices = $products;
	}

	public function render()
	{
		$this->template->setFile(__DIR__ . '/ProductsGrid.latte');
		$this->template->render();
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->remove($id))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleDeleteDuplicity()
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->deleteDuplicity())
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.duplicityProductsDeleted');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.duplicityProductsDeleteFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();
		$grid->setRememberState();
		$grid->setItemsPerPageList([20, 50, 100, 200, 500], false);
		$grid->setDefaultPerPage(50);

		foreach ($this->em->getRepository(ProductVariantSupplier::class)->createQueryBuilder('pvs')
			         ->select('pvs.quantity, IDENTITY(pv.product) as product')
			         ->join('pvs.productVariant', 'pv')->getQuery()->getArrayResult() as $row) {
			$id = $row['product'];

			if ($row['quantity'] > 0)
				$this->pvsInStock[$id] = 1;
			else if (!isset($this->pvsInStock[$id]))
				$this->pvsInStock[$id] = 0;
		}

		$qb = $this->productServices->getEr()->createQueryBuilder('p')
			->addSelect('pt, cd, cdt, cp, cpc, cpct, gal, imgs, manu, ptag, tags')
			->leftJoin('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->leftJoin('p.gallery', 'gal')
			->leftJoin('gal.images', 'imgs', 'WITH', 'imgs.isCover = 1 OR imgs.position = 0')
			->leftJoin('p.idCategoryDefault', 'cd')
			->leftJoin('cd.categoryTexts', 'cdt', 'WITH', 'cdt.lang = :lang')
			->leftJoin('p.categoryProducts', 'cp')
			->leftJoin('cp.category', 'cpc')
			->leftJoin('cpc.categoryTexts', 'cpct', 'WITH', 'cpct.lang = :lang')
			->leftJoin('p.manufacturer', 'manu')
			->leftJoin('p.productTags', 'ptag')
			->leftJoin('ptag.tag', 'tags')
			->orderBy('p.created', 'desc');
		$grid->setDataSource($qb);

		$grid->getDataSource()->onDataLoaded[] = function($items) {
			$ids      = array_map(function($row) { return $row->getId(); }, $items);
			$variants = [];

			foreach ($items as $row)
				$this->quantities[$row->getId()] = intval($row->quantity);

			if ($ids) {
				foreach ($this->em->getRepository(ProductSupplier::class)->createQueryBuilder('ps')->select('IDENTITY(ps.product) as id, ps.quantity')
					         ->where('ps.product IN (:ids)')->setParameter('ids', $ids)->getQuery()->getArrayResult() as $v)
					$this->quantities[$v['id']] += intval($v['quantity']);
				foreach ($this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')->select('IDENTITY(pv.product) as id, pv.id as variantId, pv.quantity')
					         ->where('pv.product IN (:ids)')->setParameter('ids', $ids)->getQuery()->getArrayResult() as $v) {
					$this->quantities[$v['id']] += intval($v['quantity']);
					$variants[$v['id']][]       = $v['variantId'];
				}

				foreach ($this->em->getRepository(CategoryProduct::class)->createQueryBuilder('cp')->select('ct.name, IDENTITY(cp.product) as prod')
					         ->where('cp.product IN (:prod)')
					         ->join('cp.category', 'c')->join('c.categoryTexts', 'ct', 'with', 'ct.lang = :lang')
					         ->setParameters(['prod' => $ids,
					                          'lang' => $this->translator->getLocale()])->getQuery()->getArrayResult() as $row) {
					$this->productCategories[$row['prod']][] = $row['name'];
				}

				foreach ($this->productCategories as $k => $v)
					$this->productCategories[$k] = implode(', ', $v);
			}

			foreach ($variants as $productId => $vars) {
				foreach ($this->em->getRepository(ProductVariantSupplier::class)->createQueryBuilder('pvs')->select('IDENTITY(pvs.productVariant) as id, pvs.quantity')
					         ->where('pvs.productVariant IN (:ids)')->setParameter('ids', $vars)->getQuery()->getArrayResult() as $v)
					$this->quantities[$productId] += intval($v['quantity']);
			}
		};

		// Columns
		$grid->addColumnText('image', 'eshopCatalog.defaultGrid.image', '')->setRenderer(function($row) {
			if (!$row->gallery->images || !$row->gallery->images->first())
				return '';

			return Html::el('')
				->addHtml(Html::el('img', [
					'onMouseOver' => "showPicture('spt-{$row->getId()}', 1)",
					'onMouseOut'  => "showPicture('spt-{$row->getId()}', 0)",
					'src'         => $this->imagePipe->request($row->gallery->images->first()->getFilePath(), '50x40'),
				]))
				->addHtml(Html::el('img', [
					'id'    => 'spt-' . $row->getId(),
					'class' => 'show-picture-target',
					'src'   => $this->imagePipe->request($row->gallery->images->first()->getFilePath(), '400x400'),
				]));
		})->setAlign('center');
		$grid->addColumnLink('name', 'eshopCatalog.defaultGrid.name', 'Products:edit')
			->setRenderer(function($row) {
				return Html::el('a', ['href' => $this->getPresenter()->link('Products:edit', $row->id)])
					->setText($row->getProductText()->name);
			});
		$grid->addColumnText('code1', 'code1');
		$grid->addColumnText('manufacturer', 'eshopCatalog.defaultGrid.manufacturer')->setRenderer(function($row) {
			return $row->manufacturer->name;
		});
		$grid->addColumnText('nameCategory', 'eshopCatalog.productGrid.defaultCategory')->setRenderer(function($row) {
			$text = $row->idCategoryDefault ? $row->idCategoryDefault->getCategoryText()->name : '';

			return Html::el('a', ['href' => $this->getPresenter()->link('Products:edit', $row->id)])
				->setText($text);
		});
		$grid->addColumnText('categories', 'eshopCatalog.productGrid.categories')->setRenderer(function($row) {
			return $this->productCategories[$row->getId()];
		});
		$grid->addColumnText('tags', 'eshopCatalog.productGrid.tags')->setRenderer(function($row) {
			$arr = [];
			foreach ($row->productTags->toArray() as $pt)
				$arr[] = $pt->tag->type;

			//TODO s odkazem na editaci kategorie
			return implode(', ', $arr);
		});
		$grid->addColumnNumber('price', 'eshopCatalog.productGrid.price')->setRenderer(function($row) {
			return $row->price . ' Kč';
		});
		$grid->addColumnNumber('quantity', 'eshopCatalog.productGrid.quantity')->setRenderer(function($row) {
			return $this->quantities[$row->getId()];
		});
		$grid->addColumnStatus('isPublished', 'eshopCatalog.defaultGrid.isPublished')->setAlign('center')
			->addOption(1, 'eshopCatalog.defaultGrid.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopCatalog.defaultGrid.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filters
		$grid->addFilterText('name', '')->setCondition([$this, 'conditionGridName']);
		$grid->addFilterText('code1', '');
		$grid->addFilterText('manufacturer', '')->setCondition([$this, 'conditionGridManufacturer']);
		$grid->addFilterText('nameCategory', '')->setCondition([$this, 'conditionGridCategory']);
		$grid->addFilterText('categories', '')->setCondition([$this, 'conditionGridCategories']);
		$grid->addFilterText('tags', '')->setCondition([$this, 'conditionGridTags']);
		$grid->addFilterSelect('quantity', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		])->setCondition([$this, 'conditionGridQuantity']);
		$grid->addFilterText('price', '')->setCondition([$this, 'conditionGridPrice']);
		$grid->addFilterSelect('isPublished', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		], 'p.isPublished');

		$manufacturers = [];
		foreach ($this->em->getRepository(Manufacturer::class)->createQueryBuilder('m')
			         ->select('m.id, m.name')->orderBy('m.position')->getQuery()->getArrayResult() as $m)
			$manufacturers[$m['id']] = $m['name'];

		$vatRates = [];
		foreach ($this->em->getRepository(VatRate::class)->createQueryBuilder('vr')
			         ->select('vr.id, vr.name')->getQuery()->getArrayResult() as $vr)
			$vatRates[$vr['id']] = $vr['name'];

		// Actions
		$grid->addAction('edit', '', 'Products:edit')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')->setConfirm('default.reallyDelete');

		$grid->addGroupAction('eshopCatalog.productGrid.editCategories')
			->onSelect[] = [$this, 'gridEditCategories'];
		$grid->addGroupAction('eshopCatalog.productGrid.editVariants')
			->onSelect[] = [$this, 'gridEditVariants'];
		$grid->addGroupAction('eshopCatalog.productGrid.editFeatures')
			->onSelect[] = [$this, 'gridEditFeatures'];
		$grid->addGroupSelectAction('eshopCatalog.productGrid.editManufacturer', $manufacturers)
			->onSelect[] = [$this, 'gridEditManufacturer'];
		$grid->addGroupAction('eshopCatalog.productGrid.changePublished')
			->onSelect[] = [$this, 'gridChangePublished'];
		$grid->addGroupAction('eshopCatalog.productGrid.changeUnpublished')
			->onSelect[] = [$this, 'gridChangeUnpublished'];
		$grid->addGroupSelectAction('eshopCatalog.productGrid.changeVatRate', $vatRates)
			->onSelect[] = [$this, 'gridChangeVatRate'];
		$grid->addGroupAction('eshopCatalog.productGrid.editExport')
			->onSelect[] = [$this, 'gridEditExport'];

		// Columns prototype
		$imageTdElement          = $grid->getColumn('image')->getElementPrototype('td');
		$imageTdElement->class[] = 'w1';
		$imageTdElement->style   = 'padding: 0;';

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->productServices->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridCategories(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		if ($value == '-') {
			$criteria->orWhere(Criteria::expr()->isNull('cpct'));
		} else {
			foreach (explode(',', $value) as $val)
				$criteria->orWhere(Criteria::expr()->contains('cpct.name', trim($val)));
		}

		$qb->addCriteria($criteria);
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridTags(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		if ($value == '-') {
			$criteria->orWhere(Criteria::expr()->isNull('tags'));
		} else {
			foreach (explode(',', $value) as $val)
				$criteria->orWhere(Criteria::expr()->contains('tags.type', trim($val)));
		}

		$qb->addCriteria($criteria);
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridName(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		foreach (explode(',', $value) as $val)
			$criteria->orWhere(Criteria::expr()->contains('pt.name', trim($val)));

		$qb->addCriteria($criteria);
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridPrice(QueryBuilder $qb, $value)
	{
		if (!is_numeric($value))
			return;

		$criteria = Criteria::create();
		$criteria->orWhere(Criteria::expr()->contains('p.price', $value));

		$qb->addCriteria($criteria);
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridQuantity(QueryBuilder $qb, $value)
	{
		if (!is_numeric($value))
			return;

		$arr  = [];
		$sign = $value == 1 ? '>' : '<=';

		$arr[] = '(p.quantity ' . $sign . ' 0' . ($value == 1 ? ')' : ' OR p.quantity IS NULL)');

		if (!in_array('ps', $qb->getAllAliases()))
			$qb->leftJoin('p.suppliers', 'ps');
		$arr[] = '(ps.quantity ' . $sign . ' 0' . ($value == 1 ? ')' : ' OR ps.quantity IS NULL)');

		if (!in_array('pv', $qb->getAllAliases()))
			$qb->leftJoin('p.productVariants', 'pv');
		$arr[] = '(pv.quantity ' . $sign . ' 0' . ($value == 1 ? ')' : ' OR pv.quantity IS NULL)');


		$pvsKey = $value == 1 ? 'yes' : 'no';
		$pvs    = [];
		foreach ($this->pvsInStock as $id => $stock)
			$pvs[$stock == 1 ? 'yes' : 'no'][] = $id;

		$where = implode($value == 1 ? ' OR ' : ' AND ', $arr);
		if (isset($pvs[$pvsKey]))
			$where .= ' OR p.id IN (' . implode(',', $pvs[$pvsKey]) . ')';

		$qb->andWhere($where);
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridManufacturer(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		if ($value == '-') {
			$criteria->orWhere(Criteria::expr()->isNull('manu'));
		} else {
			foreach (explode(',', $value) as $val)
				$criteria->orWhere(Criteria::expr()->contains('manu.name', trim($val)));
		}

		$qb->addCriteria($criteria);
	}

	/**
	 * @param QueryBuilder $qb
	 * @param              $value
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function conditionGridCategory(QueryBuilder $qb, $value)
	{
		$criteria = Criteria::create();

		if ($value == '-') {
			$criteria->orWhere(Criteria::expr()->isNull('cdt'));
		} else {
			foreach (explode(',', $value) as $val)
				$criteria->orWhere(Criteria::expr()->contains('cdt.name', trim($val)));
		}

		$qb->addCriteria($criteria);
	}

	/**
	 * @param array $ids
	 *
	 * @throws \Nette\Application\AbortException
	 */
	public function gridEditCategories(array $ids)
	{
		$presenter = $this->getPresenter();
		$presenter->redirect('Products:editCategories', implode('-', $ids));
	}

	/**
	 * @param array $ids
	 *
	 * @throws \Nette\Application\AbortException
	 */
	public function gridEditExport(array $ids)
	{
		$presenter = $this->getPresenter();
		$presenter->redirect('Products:editProductsExport', implode('-', $ids));
	}

	/**
	 * @param array $ids
	 *
	 * @throws \Nette\Application\AbortException
	 */
	public function gridEditVariants(array $ids)
	{
		$this->getPresenter()->redirect('Products:editVariants', implode('-', $ids));
	}

	/**
	 * @param array $ids
	 *
	 * @throws \Nette\Application\AbortException
	 */
	public function gridEditFeatures(array $ids)
	{
		$this->getPresenter()->redirect('Products:editFeatures', implode('-', $ids));
	}

	/**
	 * @param array  $ids
	 * @param string $manufacturer
	 */
	public function gridEditManufacturer(array $ids, $manufacturer)
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->setManufacturer($ids, $manufacturer)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.manufacturerChanged');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.manufacturerChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/**
	 * @param array  $ids
	 * @param string $vatRate
	 */
	public function gridChangeVatRate(array $ids, $vatRate)
	{
		$presenter = $this->getPresenter();
		if ($this->productServices->setVatRate($ids, $vatRate)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.vatRateChanged');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.vatRateChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function gridChangePublished(array $ids)
	{
		$presenter = $this->getPresenter();

		foreach ($ids as $id) {
			if (!$this->productServices->setPublish($id, 1)) {
				$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');
				break;
			}
		}

		$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function gridChangeUnpublished(array $ids)
	{
		$presenter = $this->getPresenter();


		foreach ($ids as $id) {
			if (!$this->productServices->setPublish($id, 0)) {
				$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');
				break;
			}
		}

		$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}
}
